using MicroFour.StrataFrame.UI.Collections;
using MicroFour.StrataFrame.UI.Rendering;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

namespace StrataListViewBasics
{
    public partial class Form1 : MicroFour.StrataFrame.UI.Windows.Forms.StandardForm
    {

        #region [ Constructors ]

        /// <summary>
        /// Default Constructor
        /// </summary>
        public Form1()
        {
            InitializeComponent();

            //-- Activate the first panel
            tlmMenu.SetActiveItem(0);
        }

        #endregion

        #region [ Handled Events ]

        private void lnkGrouping_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
        {
            LoadGroupingList();
        }

        private void lnkHeaderFooter_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
        {
            LoadHeaderAndFooterList();
        }

        private void bwProgress_DoWork(object sender, DoWorkEventArgs e)
        {
            //-- Establish Locals
            bool hasIncompleteItems = true;
            ProgressUpdateContents i;

            while(hasIncompleteItems)
            {
                //-- Reset the incomplete status
                hasIncompleteItems = false;

                //-- Cycle through all of the progress items
                for(int index = 0; index <= _ProgressItems.Count - 1; index++)
                {
                    //-- Get the item reference
                    i = _ProgressItems[index];

                    //-- Increment the values 
                    if(i.CurrentValue < i.MaxValue)
                    {
                        i.CurrentValue += 1;
                        hasIncompleteItems = true;
                    }

                    //-- Update the collection with the new values
                    _ProgressItems[index] = i;

                    //-- Update the progress for the item
                    bwProgress.ReportProgress(0, i);
                }

                System.Threading.Thread.Sleep(100);
            }
        }

        private void bwProgress_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            //-- Establish Locals
            ProgressUpdateContents i = (ProgressUpdateContents)e.UserState;

            //-- Find the item within the list
            foreach(StrataListViewItem li in lstProgress.Items)
            {
                if((int)li.CustomData["PrimaryKey"] == i.PrimaryKey)
                {
                    li.SubItems[1].ProgressValue = i.CurrentValue;

                    //-- If the value matches the max, then hide the progress bar again
                    if(i.CurrentValue == i.MaxValue)
                        li.SubItems[1].ProgressVisible = false;
                }
            }
        }

        private void lnkStartProgress_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
        {
            StartProgress();
        }

        private void lnkProgressList_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
        {
            LoadEmbeddedProgress();
        }

        private void lstLink_ListItemLinkClicked(MicroFour.StrataFrame.UI.Windows.Forms.EventArgs.ListItemLinkClickedEventArgs e)
        {
            //-- Determine if this is a sub item or a root row item.
            if(e.ItemType == MicroFour.StrataFrame.UI.Rendering.Enums.StrataCheckedListItemType.ListItem)
            {
                MessageBox.Show(e.Item.CustomData["PrimaryKey"].ToString() + " - " + e.Item.Text);
            }
            else
            {
                MessageBox.Show(((StrataListViewSubItem)e.Item).Parent.CustomData["PrimaryKey"].ToString() + " - " + e.Item.Text);
            }
        }


        private void lnkEmbeddedLink_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
        {
            LoadEmbeddedLink();
        }

        private void lnkSimpleSelectAll_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
        {
            foreach(StrataListViewItem i in lstSimpleCheck.Items)
            {
                i.SubItems[0].CheckedState = MicroFour.StrataFrame.UI.Rendering.Enums.StrataCheckedState.Checked;
            }
        }

        private void linkLabel1_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
        {
            foreach(StrataListViewItem i in lstSimpleCheck.Items)
            {
                i.SubItems[0].CheckedState = MicroFour.StrataFrame.UI.Rendering.Enums.StrataCheckedState.Unchecked;
            }
        }

        private void lnkViewSimpleChecked_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
        {
            //-- Establish Locals
            string msg = string.Empty;

            //-- Cycle through all of the checked items
            foreach(StrataListViewItem i in lstSimpleCheck.GetCheckedItems(1))
            {
                //-- Let's pull the custom data where we stored off the primary key of the row
                msg += i.CustomData["PrimaryKey"].ToString() + " - " + i.SubItems[0].Text + "\n";
            }

            MessageBox.Show(msg);
        }

        private void lnkSimpleChecked_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
        {
            LoadSimpleCheckList();
        }

        private void Customers_ParentFormLoading()
        {
            this.Customers.FillTop100();
        }

        private void lnkStandard_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
        {
            LoadStandardList();
        }

        #endregion

        #region [ Nested Types ]

        /// <summary>
        /// This is used for the progress updating sample.  This is only used since we are threading the sample
        /// and do not want to directly access the list view for updating the status.
        /// </summary>
        private struct ProgressUpdateContents
        {
            public int CurrentValue;
            public int MaxValue;
            public int PrimaryKey;

            public ProgressUpdateContents(int currentValue, int maxValue, int primaryKey)
            {
                CurrentValue = currentValue;
                MaxValue = maxValue;
                PrimaryKey = primaryKey;
            }
        }

        #endregion

        #region [ Private Fields ]

        private List<ProgressUpdateContents> _ProgressItems = new List<ProgressUpdateContents>();

        #endregion

        #region [ Private Methods ]

        /// <summary>
        /// Loads the list with an embedded link
        /// </summary>
        private void LoadEmbeddedLink()
        {
            //-- Establish Locals
            StrataListViewItem item;

            //-- Clear the list
            lstLink.BeginUpdate();
            lstLink.Items.Clear();

            //-- Cycle through all of the items
            foreach(CustomersBO bo in Customers.GetEnumerable())
            {
                //-- Create a new item. The item is actually the first column as well.  In this case, we do not want
                //   to show anything other than an icon.  In this example, they will all be the same and user the key "User"
                item = new StrataListViewItem(string.Empty, "User");

                //-- Create the sub items.  A sub-item is ultimately nothing more than a column.
                item.SubItems.Add(string.Format("{0}, {1}", bo.cust_LastName, bo.cust_FirstName), MicroFour.StrataFrame.UI.Rendering.Enums.StrataEmbeddedControlType.Link);
                item.SubItems.Add(string.Format("{0}, {1}, {2}  {3}", bo.cust_Address1, bo.cust_City, bo.cust_State, bo.cust_Postal), string.Empty);

                //-- Create a custom data item that stores the primary key of this record
                item.CustomData.Add("PrimaryKey", bo.cust_pk);

                //-- Add the item to the list
                lstLink.Items.Add(item);
            }

            //-- Resume rendering
            lstLink.EndUpdate();
        }

        /// <summary>
        /// Loads the list with an embedded progress bar
        /// </summary>
        private void LoadEmbeddedProgress()
        {
            //-- Establish Locals
            StrataListViewItem item;
            int oddOrEven = 0;

            //-- Clear the list
            lstProgress.BeginUpdate();
            lstProgress.Items.Clear();

            //-- Cycle through all of the items
            foreach(CustomersBO bo in Customers.GetEnumerable())
            {
                //-- Increment the odd or event indicator
                oddOrEven++;

                //-- Create a new item. The item is actually the first column as well.  In this case, we do not want
                //   to show anything other than an icon.  In this example, they will all be the same and user the key "User"
                item = new StrataListViewItem(string.Empty, "User");

                //-- Create the sub items.  A sub-item is ultimately nothing more than a column.
                item.SubItems.Add(string.Format("{0}, {1}", bo.cust_LastName, bo.cust_FirstName), string.Empty);
                item.SubItems.Add("Idle...", MicroFour.StrataFrame.UI.Rendering.Enums.StrataEmbeddedControlType.ProgressBar);

                //-- A progress bar can follow the current theme, or you can elect to show it with a unique green style.  So both
                //   can be seen in this sample, we will alternate them.
                if((oddOrEven % 2) == 0)
                    item.SubItems[1].ProgressFullThemeSupport = false;

                //-- Initially, hide the progress bar
                item.SubItems[1].ProgressVisible = false;

                //-- Create a custom data item that stores the primary key of this record
                item.CustomData.Add("PrimaryKey", bo.cust_pk);

                //-- Add the item to the list
                lstProgress.Items.Add(item);
            }

            //-- Resume rendering
            lstProgress.EndUpdate();
        }

        /// <summary>
        /// Loads the grouping list
        /// </summary>
        private void LoadGroupingList()
        {
            //-- Establish Locals
            StrataListViewItem item;

            //-- Clear the list
            lstGrouping.BeginUpdate();
            lstGrouping.Items.Clear();

            //-- Cycle through all of the items
            foreach(CustomersBO bo in Customers.GetEnumerable())
            {
                //-- Create a new item. The item is actually the first column as well.  In this case, we do not want
                //   to show anything other than an icon.  In this example, they will all be the same and user the key "User"
                item = new StrataListViewItem(string.Empty, "User");

                //-- Create the sub items.  A sub-item is ultimately nothing more than a column.
                item.SubItems.Add(string.Format("{0}, {1}", bo.cust_LastName, bo.cust_FirstName), string.Empty);
                item.SubItems.Add(string.Format("{0}, {1}, {2}  {3}", bo.cust_Address1, bo.cust_City, bo.cust_State, bo.cust_Postal), string.Empty);

                //-- Group the users by their countries.  The groups will be automatically sorted.  If you want groups to appear
                //   in the order that they were added, set the AutoGroupSort property to false.  Also in this sample, we will show
                //   the groups as collapsed initially.
                item.Group = lstGrouping.Groups[bo.cust_Country, true, true];

                //-- Create a custom data item that stores the primary key of this record
                item.CustomData.Add("PrimaryKey", bo.cust_pk);

                //-- Add the item to the list
                lstGrouping.Items.Add(item);
            }

            //-- Resume rendering
            lstGrouping.EndUpdate();
        }

        /// <summary>
        /// Loads the header and footer list
        /// </summary>
        private void LoadHeaderAndFooterList()
        {
            //-- Establish Locals
            StrataListViewItem item;
            int index = 0;

            //-- Clear the list
            lstHeaderFooter.BeginUpdate();
            lstHeaderFooter.Items.Clear();

            //-- Cycle through all of the items
            foreach(CustomersBO bo in Customers.GetEnumerable())
            {
                //-- Create a new item. The item is actually the first column as well.  
                item = new StrataListViewItem(string.Empty, string.Empty);

                //-- Create the sub items.  A sub-item is ultimately nothing more than a column.
                item.SubItems.Add(string.Format("{0}, {1}, {2}  {3}", bo.cust_Address1, bo.cust_City, bo.cust_State, bo.cust_Postal), string.Empty);
                item.SubItems.Add(bo.cust_Country, string.Empty);

                //-- Show the customer name in the header.  Show an icon on the header as well.
                item.HeaderText = string.Format("{0}, {1}", bo.cust_LastName, bo.cust_FirstName);
                item.HeaderImageKey = "User";
                
                //-- Show some additional information in the footer
                index++;
                item.FooterText = string.Format("This was item {0:n0} or {1:n0}", index, bo.Count);

                //-- Create a custom data item that stores the primary key of this record
                item.CustomData.Add("PrimaryKey", bo.cust_pk);

                //-- Add the item to the list
                lstHeaderFooter.Items.Add(item);
            }

            //-- Resume rendering
            lstHeaderFooter.EndUpdate();
        }

        /// <summary>
        /// Loads the simple check box list
        /// </summary>
        private void LoadSimpleCheckList()
        {
            //-- Establish Locals
            StrataListViewItem item;

            //-- Clear the list
            lstSimpleCheck.BeginUpdate();
            lstSimpleCheck.Items.Clear();

            //-- Cycle through all of the items
            foreach(CustomersBO bo in Customers.GetEnumerable())
            {
                //-- Create a new item. The item is actually the first column as well.  In this case, we do not want
                //   to show anything other than an icon.  In this example, they will all be the same and user the key "User"
                item = new StrataListViewItem(string.Empty, "User");

                //-- In this example, we are going to put the checkbox in the second column (which is the first sub-item)
                item.SubItems.Add(string.Format("{0}, {1}", bo.cust_LastName, bo.cust_FirstName), string.Empty, MicroFour.StrataFrame.UI.Rendering.Enums.StrataCheckBoxSupportType.TwoState, MicroFour.StrataFrame.UI.Rendering.Enums.StrataCheckedState.Unchecked);
                
                //-- Add the address
                item.SubItems.Add(string.Format("{0}, {1}, {2}  {3}", bo.cust_Address1, bo.cust_City, bo.cust_State, bo.cust_Postal), string.Empty);

                //-- Create a custom data item that stores the primary key of this record
                item.CustomData.Add("PrimaryKey", bo.cust_pk);

                //-- Add the item to the list
                lstSimpleCheck.Items.Add(item);
            }

            //-- Resume rendering
            lstSimpleCheck.EndUpdate();
        }

        /// <summary>
        /// Loads the standard list
        /// </summary>
        private void LoadStandardList()
        {
            //-- Establish Locals
            StrataListViewItem item;

            //-- Clear the list
            lstStandard.BeginUpdate();
            lstStandard.Items.Clear();

            //-- Cycle through all of the items
            foreach(CustomersBO bo in Customers.GetEnumerable())
            {
                //-- Create a new item. The item is actually the first column as well.  In this case, we do not want
                //   to show anything other than an icon.  In this example, they will all be the same and user the key "User"
                item = new StrataListViewItem(string.Empty, "User");

                //-- Create the sub items.  A sub-item is ultimately nothing more than a column.
                item.SubItems.Add(string.Format("{0}, {1}", bo.cust_LastName, bo.cust_FirstName), string.Empty);
                item.SubItems.Add(string.Format("{0}, {1}, {2}  {3}", bo.cust_Address1, bo.cust_City, bo.cust_State, bo.cust_Postal), string.Empty);

                //-- Create a custom data item that stores the primary key of this record
                item.CustomData.Add("PrimaryKey", bo.cust_pk);

                //-- Add the item to the list
                lstStandard.Items.Add(item);
            }

            //-- Resume rendering
            lstStandard.EndUpdate();
        }

        /// <summary>
        /// Prepares the progress items for use and then starts the "update" progress
        /// </summary>
        private void StartProgress()
        {
            //-- Establish Locals
            Random rand = new Random(5);
            int max;

            //-- Clear the progress items collection
            _ProgressItems.Clear();

            //-- Update each of the 
            foreach(StrataListViewItem i in lstProgress.Items)
            {
                //-- Get a random max value
                max = rand.Next(100, 200);

                //-- Create the progress item
                _ProgressItems.Add(new ProgressUpdateContents(0, max, (int)i.CustomData["PrimaryKey"]));

                //-- Reset any current value
                i.SubItems[1].ProgressValue = 0;

                //-- Go ahead and show progress bar and update the max
                i.SubItems[1].ProgressMaximum = max;
                i.SubItems[1].ProgressVisible = true;
            }

            //-- Start the update
            bwProgress.RunWorkerAsync();
        }

        #endregion
        

    }
}