﻿using System;
using System.Threading;

namespace SampleProcess
{

  /// <summary>
  /// Class that runs the process.
  /// </summary>
  public class ProcessManager
  {

    /// <summary>
    /// Raised when the step completes.
    /// </summary>
    public event EventHandler<ProcessStepCompletedEventArgs> ProcessStepCompleted;

    /// <summary>
    /// Raises the ProcessStepCompleted event.
    /// </summary>
    /// <param name="e"></param>
    protected void OnProcessStepCompleted( ProcessStepCompletedEventArgs e )
    {
      if (this.ProcessStepCompleted != null)
      {
        this.ProcessStepCompleted( this, e );
      }
    }

    /// <summary>
    /// Construct a default instance, using the default of 10 steps.
    /// </summary>
    public ProcessManager() : this( 10 ) { }

    /// <summary>
    /// Construct a new instance, setting the total number of steps.
    /// </summary>
    /// <param name="totalSteps"></param>
    public ProcessManager( int totalSteps )
    {
      this.TotalSteps = totalSteps;
    }

    /// <summary>
    /// Define the total number of steps.
    /// </summary>
    public int TotalSteps { get; set; }

    /// <summary>
    /// Validate that the provided argument for the number of steps 
    /// is valid. If not, throw exceptions.
    /// </summary>
    /// <param name="steps"></param>
    /// <returns></returns>
    public void ValidateArgs( string steps )
    {
      //-- Validate that they provided a positive number of steps.
      if (string.IsNullOrEmpty( steps ))
      {
        throw new ArgumentNullException( "Number of Steps", "You must provide the number of steps to run." );
      }
      int stepNumber;
      if (!int.TryParse( steps, out stepNumber ))
      {
        throw new ArgumentException( "You must provide an integer for the number of steps.", "Number of Steps" );
      }
      if (stepNumber < 1)
      {
        throw new ArgumentOutOfRangeException(  "Number of Steps" , "You must provide an integer greater than zero for the number of steps." );
      }
    }

    /// <summary>
    /// Run the process. This will post messages to the progress 
    /// form. 
    /// </summary>
    public void Run()
    {

      //-- Simulate a long running process. Since we interested in how
      //   to manage the process more than the process, we'll just use
      //   a random class to generate how long to put the thread to 
      //   sleep.
      Random rnd = new Random();

      for (int idx = 0 ; idx < this.TotalSteps ; idx++)
      {
        //-- Sleep somewhere between 100 ms and 1 seconds.
        int timeout = rnd.Next( 100, 1000 );
        Thread.Sleep( timeout );

        //-- Raise event that the step has completed.
        this.OnProcessStepCompleted( new ProcessStepCompletedEventArgs( idx + 1, string.Format( "{0} Step completed: {1} ms", idx + 1, timeout ) ) );
      }
    }

  }
}
