﻿using System;
using System.Collections.Generic;
using System.Threading;
using System.Windows.Forms;

namespace UnattendedModeSample
{

  /// <summary>
  /// Class that runs the process.
  /// </summary>
  class ProcessManager
  {

    /// <summary>
    /// Validate that the provided argument for the number of steps 
    /// is valid. If not, throw exceptions.
    /// </summary>
    /// <param name="steps"></param>
    /// <returns></returns>
    public void ValidateArgs( string steps )
    {
      //-- Validate that they provided a positive number of steps.
      if (string.IsNullOrEmpty( steps ))
      {
        throw new ArgumentNullException( "Number of Steps", "You must provide the number of steps to run." );
      }
      int stepNumber;
      if (!int.TryParse( steps, out stepNumber ))
      {
        throw new ArgumentException( "Number of Steps", "You must provide an integer for the number of steps." );
      }
      if (stepNumber < 1)
      {
        throw new ArgumentOutOfRangeException( "Number of Steps", "You must provide an integer greater than zero for the number of steps." );
      }
    }

    /// <summary>
    /// Run the process. This will post messages to the progress 
    /// form. This allows you pass in how many steps you want to 
    /// run.
    /// </summary>
    public void Run( int steps )
    {
      //-- Execute process, sending out messages as needed to progress form.
      ProgressForm frm = new ProgressForm();

      //-- Open the form with a initialization message.
      frm.Show();
      frm.addMessage( "Process starting..." );

      //-- Simulate a long running process. Since we interested in how
      //   to manage the process more than the process, we'll just use
      //   a random class to generate how long to put the thread to 
      //   sleep.
      Random rnd = new Random();

      for (int idx = 0 ; idx < steps ; idx++)
      {
        //-- Sleep somewhere between 100 ms and 1 seconds.
        int timeout = rnd.Next( 100, 1000 );
        Thread.Sleep( timeout );

        //-- Post message that this step is done.
        frm.addMessage( string.Format( "{0} Step completed: {1} ms", idx + 1, timeout ) );
      }

      //-- Indicate that the process is closed.
      frm.setCompleted();
    }


  }
}
