﻿using System;
using System.Collections;
using System.ComponentModel;
using System.Diagnostics;
using net = System.Windows.Forms;
using sf = MicroFour.StrataFrame.UI.Windows.Forms;

namespace LaunchDlgPropertyProvider
{

  /// <summary>
  /// Provides form launching capabilities to a select set of UI 
  /// elements.
  /// </summary>
  [ProvideProperty( "FormType", typeof( object ) )]
  public partial class LauncherProvider
    : Component, IExtenderProvider
  {

    #region [ Constructors ]

    public LauncherProvider()
    {
      InitializeComponent();
    }

    public LauncherProvider( IContainer container )
    {
      container.Add( this );

      InitializeComponent();
    }

    #endregion [ Constructors ]

    #region [ Extended Properties ]

    private Hashtable _formTypeHash = new Hashtable();

    /// <summary>
    /// Get the form type for an extended control.
    /// </summary>
    [Category( "Form Launch Setup" )]
    [DisplayName( "FormType" )]
    [Description( "The fully qualified class name of the form to open." )]
    public String GetFormType( object extendee )
    {
      return ExtenderProvider.GetProperty<string>( extendee, this._formTypeHash, string.Empty );
    }

    /// <summary>
    /// Set the FormType for a control.
    /// </summary>
    public void SetFormType( object extendee, object value )
    {
      //-- Determine if the value is already set.
      bool isAlreadySet = !string.IsNullOrEmpty( this.GetFormType( extendee ) );

      //-- If the value was already setup and is null or empty string, 
      //   remove the handler. Unfortunately, we have to handle the 
      //   toolstripbutton differently, since it doesn't inherit from 
      //   Control. Note we are using a ToolStripItem so we can extend 
      //   to other types of toolstrip items if desired, including menu 
      //   items.
      if (isAlreadySet && (value == null || string.IsNullOrEmpty( (string)value )))
      {
        if (extendee is net.Control)
        {
          ((net.Control)extendee).Click -= this.Item_Click;
        }
        else
        {
          ((net.ToolStripItem)extendee).Click -= this.Item_Click;
        }
      }

      //-- Set form type.
      ExtenderProvider.SetProperty<string>( extendee, this._formTypeHash, value );

      //-- If value is being set (and not changed), add a handler. 
      if (!isAlreadySet && value != null && !string.IsNullOrEmpty( (string)value ))
      {
        if (extendee is net.Control)
        {
          ((net.Control)extendee).Click += this.Item_Click;
        }
        else
        {
          ((net.ToolStripItem)extendee).Click += this.Item_Click;
        }
      }
    }

    #endregion [ Extended Properties ]

    #region IExtenderProvider Members

    /// <summary>
    /// Determine if a control (the extendee) can be extended by this
    /// provider.
    /// </summary>
    /// <remarks>
    /// This will extend control that inherit from button, link and  
    /// toolstrip button.
    /// </remarks>
    public bool CanExtend( object extendee )
    {
      //-- Establish return var.
      bool isExtendable = false;

      //-- Test if control is one of the extendable types.
      if (extendee is net.Button) isExtendable = true;
      else if (extendee is net.ToolStripButton) isExtendable = true;
      else if (extendee is net.LinkLabel) isExtendable = true;

      //-- Return if the control is extendable.
      return isExtendable;
    }

    #endregion

    #region [ Event Handlers ]

    /// <summary>
    /// Handle click of extended controls to launch form.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    private void Item_Click( object sender, EventArgs e )
    {
      //-- Lookup the form type. The sender is the control being
      //   clicked, so we just needt to lookup form type for that
      //   control.
      string formTypeName = this.GetFormType( sender );
      Type formType = Type.GetType( formTypeName );

      //-- Launch the form.
      Launcher.LaunchDialog( formType );
    }

    #endregion [ Event Handlers ]


  }
}
