﻿using System;
using System.Collections.Generic;
using System.Reflection;
using System.Windows.Forms;

namespace ProfilerExample
{

  /// <summary>
  /// Provides a simple message box based logger for the profiler. 
  /// This is used when debugging the profiler code itself.
  /// </summary>
  class DebugProfileLogger
    : IProfileLogger
  {
    #region [ IProfileLogger Members ]

    public MethodBase Method { get; set; }

    public string CodeBlockName { get; set; }

    public void EnterMethod()
    {
      if (!this.ValidateRequiredData())
      {
        throw new InvalidOperationException( "The required data is not set for logging. Set the Method property and the CodeBlockName." );
      }

      //-- Log that we are entering the method/block.
      MessageBox.Show( string.Format( "ENTERING: {0}", this.Method.Name ), string.Format( "Type: {0}", this.Method.DeclaringType.FullName ) );
    }

    public void ExitMethod( TimeSpan elapsedTime )
    {
      //-- Log that we are exiting the method/block.
      MessageBox.Show( string.Format( "EXITING: {0} [{1} seconds]", this.Method.Name, elapsedTime.TotalSeconds ), string.Format( "Type: {0}", this.Method.DeclaringType.FullName ) );
    }

    /// <summary>
    /// Show the mark
    /// </summary>
    /// <param name="markName"></param>
    /// <param name="elapsedTime"></param>
    public void Mark( string markName, TimeSpan elapsedTime )
    {
      MessageBox.Show( string.Format( "  {2}: {0} [{1} seconds]", this.Method.Name, elapsedTime.TotalSeconds, markName ), string.Format( "Type: {0}", this.Method.DeclaringType.FullName ) );
    }

    #endregion [ IProfileLogger Members ]

    #region [ Private (helper) Methods ]

    /// <summary>
    /// Validate the logger is setup for logging.
    /// </summary>
    /// <returns></returns>
    private bool ValidateRequiredData()
    {
      return this.Method != null && !string.IsNullOrEmpty( this.CodeBlockName );
    }

    #endregion [ Private (helper) Methods ]

  }
}
