﻿using System;
using System.Collections.Generic;
using System.Reflection;
using System.Diagnostics;

namespace ProfilerExample
{

  /// <summary>
  /// Provides profiling services to the application.
  /// </summary>
  /// <remarks>
  /// Usage:
  /// 
  /// In the main method of the application, setup the profiler. 
  /// The minimum is to set the type of logger to use:
  /// 
  /// Profiler.LoggerType = typeof(LoggerType);
  /// 
  /// You can also enable/disable the profiler:
  /// 
  /// Profiler.Enabled = true;
  /// 
  /// You can also enable the profiler via command line arguments:
  /// 
  /// Profiler.ProcessArgs(args);
  /// 
  /// This allows you to selectively enable profiling on a
  /// client-by-client basis using a command line argument. It 
  /// supports two method of turning on profiling. The first is to 
  /// simply turn it on for all types:
  /// 
  /// myapp.exe --EnableProfiling
  /// 
  /// The second way to so enable it for selected types:
  /// 
  /// myapp.exe 
  /// --EnableProfiling:Myapp.Tools.Crypto,Myapp.UI.Messages
  /// 
  /// The enable profiling indicator is followed by a comma 
  /// delimited list of the fully qualified type name of any type 
  /// you wish to profile (you must still add the profiling code to
  /// those types though).  The enable profiling indicator 
  /// (--EnableProfiling in the examples above) and the delimiter 
  /// between this indicator and the list of types are both 
  /// configurable.
  /// 
  /// Once you have it setup, you must add code to profile your 
  /// existing code. The profiler uses the IDisposable pattern to 
  /// manage the profiling. Around any code that you want to 
  /// profile, add a using statement similar to this:
  /// 
  /// using (Profiler profile = new
  ///         Profiler(System.Reflection.MethodBase.GetCurrentMethod()))
  /// {
  ///   //-- profiled code here
  /// }
  /// 
  /// You can profile an entire method (by putting the using block 
  /// around all the code in a method) or just a sub set (e.g. put 
  /// it around a loop). You can also nest profiles.
  /// 
  /// Finally, you can add a "mark" within a profile. This can be 
  /// used to mark when a certain point in the code is reached.
  /// 
  /// The output is controlled by a logger. The logger must 
  /// implement the IProfileLogger interface. This allows for the 
  /// profile output to be written to a file, stored in a database,
  /// shown in a form, uploaded to a web site via a web service, 
  /// etc. If the logger implements IDisposable, then the Dispose 
  /// method is called (to close files, clean up db connections 
  /// etc.).
  /// 
  /// This sample includes three loggers. One shows the messages 
  /// via a MessageBox and was used to debug the profiler itself.
  /// Then there are two form based loggers. One loads a textbox
  /// with the messages and one loads a listview.
  /// 
  /// When profiling, the time it takes for the logging is not 
  /// included. However, if you nest profilers, then the outer 
  /// profilers will include the time of the inner profilers. For 
  /// this reason, it is a good idea to add marks before and after 
  /// calls to other methods that are profiled. 
  /// </remarks>
  public class Profiler
    : IDisposable
  {

    #region [ Constructors ]

    /// <summary>
    /// Create an instance using the indicated method.
    /// </summary>
    /// <param name="method"></param>
    public Profiler( MethodBase method ) : this( method, "Full Method" ) { }

    /// <summary>
    /// Create an instance using the indicated method and providing a
    /// name for the block being profiled.
    /// </summary>
    /// <param name="method">MethodBase object describing method 
    /// profiled code is contained within.</param>
    /// <param name="blockName">name for the code being profiled, 
    /// used when only a portion of the code in a method is being 
    /// profiled.</param>
    public Profiler( MethodBase method, string blockName )
    {
      //-- Validate that this method can be profiled.
      if (!Profiler.GetMethodPermission( method )) return;

      //-- Store away the method for later use.
      this.Method = method;

      //-- Initialize logger for this profile.
      this.Logger = (IProfileLogger)System.Activator.CreateInstance( Profiler.LoggerType );
      this.Logger.Method = method;
      this.Logger.CodeBlockName = blockName;

      //-- Log that the block has been entered.
      this.Logger.EnterMethod();

      //-- Initialize the stop watch and start it. This is done
      //   here so the initialization of the logger doesn't impact
      //   the profile.
      this.Watch = Stopwatch.StartNew();
    }

    #endregion [ Constructors ]

    #region [ Static Properties ]

    /// <summary>
    /// Defines if profiling is turned on or off.
    /// </summary>
    public static bool Enabled { get; set; }

    /// <summary>
    /// Defines the profile logger type used with the profiler.
    /// </summary>
    /// <remarks>
    /// This is set for all instances of the profiler, so it only has
    /// to be initialized once. This allows the profiler to 
    /// initialize an instance of the logger in the constructor.
    /// </remarks>
    public static Type LoggerType { get; set; }

    private static List<string> _profiledMethods;

    /// <summary>
    /// Returns a list of methods that are being profiled.
    /// </summary>
    /// <value></value>
    /// <returns></returns>
    /// <remarks></remarks>     
    public static List<string> ProfiledMethods
    {
      get
      {
        if (_profiledMethods == null)
        {
          _profiledMethods = new List<string>();
        }
        return _profiledMethods;
      }
    }

    private static string _enableProfileArgument = "--EnableProfiling";

    /// <summary>
    /// Define the argument used to turn on profiling via the command
    /// line.
    /// </summary>
    public static string EnableProfileArgument
    {
      get
      {
        return _enableProfileArgument;
      }
      set
      {
        _enableProfileArgument = value;
      }
    }

    private static string _profiledMethodDelimiter = ":";

    /// <summary>
    /// Defines the delimiter between the list of methods to profile 
    /// and the enable profiling argument indicator.
    /// </summary>
    /// <value></value>
    /// <returns></returns>
    /// <remarks></remarks>     
    public static string ProfiledMethodDelimiter
    {
      get
      {
        return _profiledMethodDelimiter;
      }
      set
      {
        _profiledMethodDelimiter = value;
      }
    }

    #endregion [ Static Properties ]

    #region [ Static Methods ]

    /// <summary>
    /// Process the arguemnts provided to the application. This 
    /// allows for the profiler to be normally turned off, but then 
    /// to be turned back on for specific clients via command line 
    /// arguments.
    /// </summary>
    /// <param name="args">command line arguments</param>
    /// <remarks>
    /// This can globally turn on processing or turn it on for one or
    /// more specific types.
    /// </remarks>
    public static void ProcessArgs( string[] args )
    {
      //-- Return if no arguments are provided.
      if (args.Length == 0) return;

      //-- Clear out current list of profiled types.
      Profiler.ProfiledMethods.Clear();

      //-- Check if the EnableProfiling argument is provided.
      bool profilingEnabled = false;
      string profileArg = string.Empty;
      for (int idx = 0 ; idx < args.Length ; idx++)
      {
        profilingEnabled = args[idx].Substring( 0, Profiler.EnableProfileArgument.Length ) == Profiler.EnableProfileArgument;
        profileArg = args[idx];
        if (profilingEnabled) break;
      }

      //-- Check if specific methods have been provided.
      if (profilingEnabled && profileArg.IndexOf( Profiler.ProfiledMethodDelimiter ) >= 0)
      {
        //-- Extra list of profiled types.
        string types = profileArg.Substring( Profiler.EnableProfileArgument.Length + 1 );

        //-- Add each of these types to the list of profiled types.
        Profiler.ProfiledMethods.AddRange( types.Split( ',' ) );
      }

      //-- Turn on profiling if indicated.
      Profiler.Enabled = profilingEnabled;

    }

    /// <summary>
    /// Return if the indicated method can be profiled. This takes 
    /// into consideration two factors: if profiling is enabled and 
    /// then if the types being profiled are being limited. If 
    /// profiling is not enabled, then nothing is profiled. If it is 
    /// enabled, then we also check if the indicated method belongs 
    /// to a type that is being profiled. If there are not types 
    /// specified as being profiled, then all types are assumed.
    /// </summary>
    /// <param name="method"></param>
    /// <returns></returns>
    public static bool GetMethodPermission( MethodBase method )
    {
      //-- Assume that if there is no method, they don't have permission.
      if (method == null) return false;

      //-- Establish return var
      bool canProfileMethod = Profiler.Enabled;

      //-- If it is enabled, see if it is turned on for this specific method.
      if (Profiler.Enabled && Profiler.ProfiledMethods.Count > 0)
      {
        canProfileMethod = Profiler.ProfiledMethods.Contains( method.DeclaringType.FullName );
      }

      //-- Return if this method can be profiled.
      return canProfileMethod;
    }

    #endregion [ Static Methods ]

    #region [ Properties ]

    /// <summary>
    /// Define the method being logged.
    /// </summary>
    public MethodBase Method { get; set; }

    /// <summary>
    /// Defines an instance of the logger used for the profile.
    /// </summary>
    public IProfileLogger Logger { get; set; }

    #region [  Private Properties ]

    /// <summary>
    /// Defines a stop watch that is used to time how long a section 
    /// of code takes to run.
    /// </summary>
    private Stopwatch Watch { get; set; }

    #endregion [  Private Properties ]

    #endregion [ Properties ]

    #region [ Methods ]

    /// <summary>
    /// Add a profile mark to the profile. This is used within a 
    /// profile to mark specific sections of the code.
    /// </summary>
    /// <param name="markName"></param>
    public void AddMark( string markName )
    {
      //-- Validate that this method can be profiled.
      if (!Profiler.GetMethodPermission( this.Method )) return;

      //-- Stop timer so logging doesn't affect profile.
      this.Watch.Stop();

      //-- Log time.
      this.Logger.Mark( markName, this.Watch.Elapsed );

      //-- Restart stop watch.
      this.Watch.Start();
    }

    #endregion [ Methods ]

    #region [ IDisposable Members ]

    /// <summary>
    /// Call profiling code to log exit from profiler and provide 
    /// the elapsed time.
    /// </summary>
    public void Dispose()
    {
      //-- Validate that this method was being profiled.
      if (!Profiler.GetMethodPermission( this.Method )) return;

      //-- Stop timing immediately, so logging doesn't impact the
      //   profile.
      this.Watch.Stop();

      //-- Log that the code is exited.
      this.Logger.ExitMethod( this.Watch.Elapsed );

      //-- If the logger implement IDisposable, dispose of it.
      if (this.Logger is IDisposable)
      {
        ((IDisposable)this.Logger).Dispose();
      }
    }

    #endregion [ IDisposable Members ]

  }
}
