﻿using System;
using System.Windows.Forms;

namespace ProfilerExample
{

  /// <summary>
  /// Manage implementing a special cursor during a process. A typically
  /// scenario is that you want to have the wait cursor show while running
  /// some process, then put the cursor back to what it was.
  ///
  /// The ClearCursor and SetTempCursor allow for this to be used within
  /// a process that requires the cursor to be set/cleared multiple times
  /// such as when using a dialog to collect information.
  /// </summary>
  /// <remarks>
  /// The first example will set the cursor to the AppStarting cursor and
  /// then return it to whatever it was when the using statement was called.
  ///
  /// <code>
  /// Using keeper As New CursorKeeper(Cursor.AppStarting)
  /// ' do something here
  /// End Using
  /// </code>
  ///
  /// The second example will set the cursor to the WaitCursor (the default).
  /// <code>
  /// Using keeper As New CursorKeeper()
  /// ' do something here
  /// End Using
  /// </code>
  /// 
  /// The third example will set the cursor, then while 
  /// calling a dialog, clear it, then reinstate it.
  /// <code>
  /// Using keeper As New CursorKeeper()
  /// ' Initial process....
  /// ' Clear cursor (back to original) to open dialog 
  /// keeper.ClearCursor()
  /// 'open dialog, get data...
  /// ' Reset to temp cursor
  /// keeper.SetTempCursor()
  /// ' Finish up work...
  /// End Using
  /// </code>
  /// </remarks>
  public class CursorKeeper
    : IDisposable
  {

    #region [ Constructors ]

    /// <summary>
    /// Construct a default instance using the standard wait 
    /// cursor.
    /// </summary>
    public CursorKeeper()
      : this(Cursors.WaitCursor)
    { }

    /// <summary>
    /// Construct an instance using the indicated cursor as the 
    /// temporary cursor.
    /// </summary>
    /// <param name="tempCursor"></param>
    public CursorKeeper(Cursor tempCursor)
    {
      this.OriginalCursor = Cursor.Current;
      this.TempCursor = tempCursor;
      Cursor.Current = tempCursor;
    }

    #endregion [ Constructors ]

    #region [ Properties ]

    private Cursor _originalCursor;

    /// <summary>
    /// Define the original cursor, that will be reinstated when 
    /// the instance is disposed of.
    /// </summary>
    /// <value></value>
    /// <returns></returns>
    /// <remarks></remarks>     
    public Cursor OriginalCursor
    {
      get
      {
        return _originalCursor;
      }
      set
      {
        _originalCursor = value;
      }
    }

    private Cursor _tempCursor;

    /// <summary>
    /// Define the temporary cursor. 
    /// </summary>
    /// <value></value>
    /// <returns></returns>
    /// <remarks></remarks>     
    private Cursor TempCursor
    {
      get
      {
        return _tempCursor;
      }
      set
      {
        _tempCursor = value;
      }
    }

    #endregion [ Properties ]

    #region [ Methods ]

    /// <summary>
    /// Clear the cursor back to the original cursor.
    /// </summary>
    /// <remarks></remarks>
    public void ClearCursor()
    {
      Cursor.Current = this.OriginalCursor;
    }

    /// <summary>
    /// Set the current cursor to the temp cursor.
    /// </summary>
    /// <remarks></remarks>
    public void SetTempCursor()
    {
      Cursor.Current = this.TempCursor;
    }

    #endregion [ Methods ]

    #region [ IDisposable Members ]

    private bool _disposedValue = false; //-- Detect redundant calls.

    /// <summary>
    /// Implement a dispose method that checks if instance is already
    /// being disposed of.
    /// </summary>
    /// <param name="disposing"></param>
    /// <remarks>
    /// Here is where the original cursor is reinstated. This
    /// allows the using statement to used to easily restore
    /// the original cursor.
    /// </remarks>
    protected virtual void Dispose(bool disposing)
    {
      if (!this._disposedValue)
      {
        if (disposing)
        {
          //-- Reinstate the original cursor
          Cursor.Current = this.OriginalCursor;
        }
      }
      this._disposedValue = true;
    }

    /// <summary>
    /// Implement the Dispose Method, coded to correctly implement the
    /// disposable pattern.
    /// </summary>
    /// <remarks></remarks>
    public void Dispose()
    {
      //-- Do not change this code.  Put cleanup code in Dispose(ByVal disposing As Boolean) above.
      Dispose(true);
      GC.SuppressFinalize(this);
    }

    #endregion [ IDisposable Members ]

  }
}
