﻿using System;
using System.Reflection;
using System.Diagnostics;

namespace ProfilerExample
{

  /// <summary>
  /// Define interface for loggers used with the Profiler class.
  /// </summary>
  /// <remarks>
  /// When implementing, the methods should ensure that at least 
  /// the Method property is set, as this is how the profiler 
  /// passes details about the method to the logger. The 
  /// CodeBlockName should be optional, as if an entire method is 
  /// being profiled, there is not need for it.
  /// 
  /// While it isn't enforced, the logger must have a parameterless
  /// constructor.
  /// </remarks>
  public interface IProfileLogger
  {
    /// <summary>
    /// Defines the method containing the code being profiled.
    /// </summary>
    MethodBase Method { get; set; }

    /// <summary>
    /// Defines a name for a code block. This is only needed if a 
    /// sub-block of code within a method is being profiled.
    /// </summary>
    string CodeBlockName { get; set; }

    /// <summary>
    /// Log that a block of code is being entered.
    /// </summary>
    void EnterMethod();

    /// <summary>
    /// Log that a block of code is being exited.
    /// </summary>
    /// <param name="elapsedTime">elapsed time since profiling 
    /// started</param>
    void ExitMethod( TimeSpan elapsedTime );

    /// <summary>
    /// Log a note to note intermediate steps within a block of code. 
    /// </summary>
    /// <param name="markName">Name used to reference this place in 
    /// code</param>
    /// <param name="elapsedTime">elapsed time since profiling 
    /// started</param>
    void Mark(string markName, TimeSpan elapsedTime );
  }
}
