using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Reflection;
using System.Windows.Forms;
using sf = MicroFour.StrataFrame.UI.Windows.Forms;
using MicroFour.StrataFrame.Data;

namespace ConcurrencySample
{
  public partial class MainForm : MicroFour.StrataFrame.UI.Windows.Forms.StandardForm
  {

    #region [ Constructors ]

    public MainForm()
    {
      InitializeComponent();
      this.chkAutoHandleCollisions.Checked = this.AutoHandleCollisions;
      this.SqlDebugFile = Path.Combine( Path.GetDirectoryName( Assembly.GetExecutingAssembly().Location ), "CollisionSql.html" );
      DataLayer.DataSources[""].SetDebugOn( this.SqlDebugFile, true );
      this.chkDebugSql.Checked = true;
      this.lnkSqlDebugFile.Text = this.SqlDebugFile;
    }

    #endregion [ Constructors ]

    #region [ Properties ]

    /// <summary>
    /// Full path to the SQL debug file.
    /// </summary>
    private string SqlDebugFile { get; set; }

    #endregion [ Properties ]

    #region [ BO Event Handlers ]

    private void customersBO1_ParentFormLoading()
    {
      this.customersBO1.FillTopN( 10 );
    }

    private void customersWithDescBO1_ParentFormLoading()
    {
      this.customersWithDescBO1.FillTopN( 10 );
    }

    private void customersBO1_ConcurrencyException( MicroFour.StrataFrame.Data.ConcurrencyExceptionEventArgs e )
    {
      this.HandleConcurrency( e, this.customersBO1 );
    }

    /// <summary>
    /// Manipulate the FieldCollisionCollection to remove system 
    /// fields, then call the standard collision form.
    /// </summary>
    /// <param name="e"></param>
    private void customersWithDescBO1_ConcurrencyException( MicroFour.StrataFrame.Data.ConcurrencyExceptionEventArgs e )
    {
      this.HandleConcurrency( e, this.customersWithDescBO1 );
    }

    #endregion [ BO Event Handlers ]

    #region [ Right Side Toolstrip Handlers ]

    private void tsbNew_Click( object sender, EventArgs e )
    {
      this.customersBO1.Add();
    }

    private void tsbEdit_Click( object sender, EventArgs e )
    {
      this.customersBO1.Edit();
    }

    private void tsbDelete_Click( object sender, EventArgs e )
    {
      this.customersBO1.DeleteCurrentRow();
    }

    private void tsbSave_Click( object sender, EventArgs e )
    {
      this.customersBO1.Save();
    }

    private void tsbUndo_Click( object sender, EventArgs e )
    {
      this.customersBO1.Undo( MicroFour.StrataFrame.Business.BusinessUndoType.CurrentRowOnly );
    }

    private void tsbFirst_Click( object sender, EventArgs e )
    {
      this.customersBO1.Navigate( MicroFour.StrataFrame.Business.BusinessNavigationDirection.First );
    }

    private void tsbPrev_Click( object sender, EventArgs e )
    {
      this.customersBO1.Navigate( MicroFour.StrataFrame.Business.BusinessNavigationDirection.Prior );
    }

    private void tsbNext_Click( object sender, EventArgs e )
    {
      this.customersBO1.Navigate( MicroFour.StrataFrame.Business.BusinessNavigationDirection.Next );
    }

    private void tsbLast_Click( object sender, EventArgs e )
    {
      this.customersBO1.Navigate( MicroFour.StrataFrame.Business.BusinessNavigationDirection.Last );
    }

    #endregion [ Right Side Toolstrip Handlers ]

    private void HandleConcurrency( MicroFour.StrataFrame.Data.ConcurrencyExceptionEventArgs e, CustomerBaseBO bo )
    {
      //-- Skip if main form is handling collisions.
      if (this.AutoHandleCollisions) return;

      //-- Establish locals
      sf.SoftCollisionDialog collisionForm;

      //--Determine if the form needs to be shown
      //--Create the form 
      collisionForm = new sf.SoftCollisionDialog();

      //-- Separate the system fields from the User fields.
      FieldCollisionCollection sysCollisions = bo.GetSystemCollisions( e.Collision );
      FieldCollisionCollection userCollisions = bo.GetUserCollisions( e.Collision );

      //-- Clear current collisions so we can show only user collisions to user
      //   and still add system collisions back in before saving.
      e.Collision.FieldCollisions.Clear();

      //-- If there are user collisions, show the collision form so the user
      //   can resolve the issues. Default to true, to handle case of 
      //   no user collisions, but there would be system collisions, 
      //   which we want to handle automatically.
      bool resave = true;
      if (userCollisions.Count > 0)
      {
        //-- Redo row collision with just user field collisions.
        e.Collision.FieldCollisions.AddRange( userCollisions.ToArray() );

        //--Set the collision on the form
        collisionForm.Collision = e.Collision;

        resave = collisionForm.ShowDialog() == System.Windows.Forms.DialogResult.OK;
      }

      //-- If indicated, resave record.
      if (resave)
      {
        //-- Rebuild collisions collection. This already has the user 
        //   collisions, so just add the system collisions.
        e.Collision.FieldCollisions.AddRange( sysCollisions.ToArray() );

        //-- Resave record
        e.ResaveRecord = true;
      }
      else
      {
        e.ResaveRecord = false;
      }
    }

    private void chkAutoHandleCollisions_CheckedChanged( object sender, EventArgs e )
    {
      this.AutoHandleCollisions = this.chkAutoHandleCollisions.Checked;
    }

    private void chkDebugSql_CheckedChanged( object sender, EventArgs e )
    {
      if (this.chkDebugSql.Checked)
      {
        DataLayer.DataSources[""].SetDebugOn( this.SqlDebugFile, true );
      }
      else
      {
        DataLayer.DataSources[""].SetDebugOff();
      }
    }

    /// <summary>
    /// Open the SQL debug file.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    private void lnkSqlDebugFile_LinkClicked( object sender, LinkLabelLinkClickedEventArgs e )
    {
      //-- Open file in default browser.
      System.Diagnostics.Process.Start( this.SqlDebugFile );
    }

    private void tsbRightSynch_Click( object sender, EventArgs e )
    {
      //-- Do nothing if either side is not idle.
      if (this.customersBO1.EditingState != MicroFour.StrataFrame.Business.BusinessEditingState.Idle
          && this.customersWithDescBO1.EditingState != MicroFour.StrataFrame.Business.BusinessEditingState.Idle)
      {
        MessageBox.Show( "Cannot synch if either side is editing." );
        return;
      }

      //-- Copy the data from right into left.
      this.customersWithDescBO1.CopyDataFrom( this.customersBO1, MicroFour.StrataFrame.Business.BusinessCloneDataType.ClearAndFillFromCompleteTable );

    }

    private void tsbLeftSynch_Click( object sender, EventArgs e )
    {
      //-- Do nothing if either side is not idle.
      if (this.customersBO1.EditingState != MicroFour.StrataFrame.Business.BusinessEditingState.Idle
          && this.customersWithDescBO1.EditingState != MicroFour.StrataFrame.Business.BusinessEditingState.Idle)
      {
        MessageBox.Show( "Cannot synch if either side is editing." );
        return;
      }

      //-- Copy the data from left into right
      this.customersBO1.CopyDataFrom( this.customersWithDescBO1, MicroFour.StrataFrame.Business.BusinessCloneDataType.ClearAndFillFromCompleteTable );
    }


  }
}