using Microsoft.VisualBasic;
using System;
using System.ComponentModel;
using MicroFour.StrataFrame.Business;


namespace StrataFlix.UI
{
	namespace Windows
	{
		namespace Forms
		{
			
			
			/// <summary>
			/// The base form used within the application
			/// </summary>
			/// <remarks></remarks>
			public class ApplicationBaseForm : MicroFour.StrataFrame.UI.Windows.Forms.StandardForm
			{
				
				#region " Private Fields "
				
				private ActionLinkMenuItemCollection _ActionItems = new ActionLinkMenuItemCollection();
				private bool _IsModified = false;
				private ActionLinkMenuItem _SaveAction;
				private ActionLinkMenuItem _SaveUndoSep;
				private ActionLinkMenuItem _UndoAction;
                private bool _AreSaveUndoActionsCreated = false;
				
				#endregion
				
				#region " Private Methods "
				
				/// <summary>
				/// Handles the IsDirtyChanged event of a BO
				/// </summary>
				/// <param name="sender"></param>
				/// <param name="e"></param>
				/// <remarks></remarks>
				private void HandleDirtyChanged(object sender, EventArgs e)
				{
					//-- The visibility of the save and undo buttons needs to change
					//   Cycle through all of the business objects... if one of them is dirty, then
					//   the save and undo need to be visible
					foreach (BusinessLayer loBO in this.BusinessObjects)
					{
						if (loBO.IsDirty)
						{
							this.SetModified(true);
							return;
						}
					}
					
					//-- If we reach here, then none of the business objects were dirty
					this.SetModified(false);
				}
				
				/// <summary>
				/// Saves the business objects within the dialog
				/// </summary>
				/// <remarks></remarks>
				private void SaveClicked()
				{
					if (Save() == MicroFour.StrataFrame.Data.SaveUndoResult.Success)
					{
						SetModified(false);
					}
				}
				
				/// <summary>
				/// Sets this form to the specified "modified" state.
				/// </summary>
				/// <param name="IsModified"></param>
				/// <remarks></remarks>
				private void SetModified(bool IsModified)
				{
					//-- If the modified state equals the current modified state, then bail
					if (this._IsModified == IsModified)
					{
						return;
					}
					
					//-- Update any object states
					SetObjectStates();
					
					//-- Save off the value
					this._IsModified = IsModified;
					
					//-- Set the visibility of the actions
					if (_ActionItems.IndexOf(_SaveUndoSep) > 0)
					{
						this._SaveUndoSep.Visible = IsModified;
					}
					else
					{
						this._SaveUndoSep.Visible = false;
					}
					
					this._SaveAction.Visible = IsModified;
					this._UndoAction.Visible = IsModified;
				}
				
				/// <summary>
				/// Undoes the business objects within the dialog
				/// </summary>
				/// <remarks></remarks>
				private void UndoClicked()
				{
					if (Undo() == MicroFour.StrataFrame.Data.SaveUndoResult.Success)
					{
						SetModified(false);
					}
				}
				
				#endregion
				
				#region " Protected Methods "
				
				/// <summary>
				/// Add handlers so that additional logic can take place when a BO becomes dirty
				/// </summary>
				/// <param name="BusinessObject"></param>
				/// <param name="AddEventHandlers"></param>
				/// <remarks></remarks>
				public override void AddBusinessObject(MicroFour.StrataFrame.Business.BusinessLayerBase BusinessObject, bool AddEventHandlers)
				{
					base.AddBusinessObject(BusinessObject, AddEventHandlers);
					
					//-- Add a handler to monitor the IsDirtyChanged event
					((MicroFour.StrataFrame.Business.BusinessLayer) BusinessObject).IsDirtyChanged += new System.EventHandler(HandleDirtyChanged);
				}

                /// <summary>
                /// Add the base save and undo actions that will be available within all dialogs
                /// </summary>
                /// <param name="e"></param>
                /// <remarks></remarks>
                protected override void OnActivated(EventArgs e)
                {
                    if ((!this.DesignMode) && (!_AreSaveUndoActionsCreated))
                    {
                        //-- Create the core logic actions
                        this._SaveUndoSep = new ActionLinkMenuItem();
                        _SaveUndoSep.ItemType = MicroFour.StrataFrame.UI.Windows.Forms.LinkMenuItemType.Separator;
                        _SaveUndoSep.Visible = false;

                        //-- Create the save and undo actions
                        _SaveAction = new ActionLinkMenuItem("Save", "Save", StrataFlix.UI.Properties.CommonImages.Save_16, new StrataFlix.UI.Windows.Forms.ActionLinkMenuItem.ClickHandler(SaveClicked), false);
                        _UndoAction = new ActionLinkMenuItem("Undo", "Undo", StrataFlix.UI.Properties.CommonImages.Undo_16, new StrataFlix.UI.Windows.Forms.ActionLinkMenuItem.ClickHandler(UndoClicked), false);

                        //-- Add the actions to the collection
                        _ActionItems.Add(_SaveUndoSep);
                        _ActionItems.Add(_SaveAction);
                        _ActionItems.Add(_UndoAction);
                    }

                    //-- Call the base logic
                    base.OnActivated(e);
                }
				
				
				/// <summary>
				/// Render the background as a themed solid color
				/// </summary>
				/// <param name="e"></param>
				/// <remarks></remarks>
				protected override void OnPaintBackground(System.Windows.Forms.PaintEventArgs e)
				{
					if (MicroFour.StrataFrame.UI.Windows.Forms.ApplicationTheme.ThemeProperties != null)
					{
						e.Graphics.Clear(MicroFour.StrataFrame.UI.Windows.Forms.ApplicationTheme.ThemeProperties.SolidBackground);
					}
					else
					{
						base.OnPaintBackground(e);
					}
				}
				
				/// <summary>
				/// Remove handlers specific to the BO
				/// </summary>
				/// <param name="BusinessObject"></param>
				/// <param name="RemoveHandlers"></param>
				/// <remarks></remarks>
				public override void RemoveBusinessObject(MicroFour.StrataFrame.Business.BusinessLayer BusinessObject, bool RemoveHandlers)
				{
					base.RemoveBusinessObject(BusinessObject, RemoveHandlers);
					
					//-- Remove the handler for the BO so that it will no longer be monitored
					BusinessObject.IsDirtyChanged -= new System.EventHandler(HandleDirtyChanged);
				}
				
				/// <summary>
				/// Sets the object states within a dialog.  This would include such items as
				/// setting a toolstrip item enabled state, etc.
				/// </summary>
				/// <remarks></remarks>
				protected virtual void SetObjectStates()
				{
					
				}
				#endregion
				
				#region " Public Properties "
				
				/// <summary>
				/// The action items that are avaialble within this dialog that will appear within the main dialog
				/// window.
				/// </summary>
				[Category("StrataFlix: Actions")][DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]public ActionLinkMenuItemCollection ActionItems
				{
					get
					{
						return this._ActionItems;
					}
					set
					{
						this._ActionItems = value;
					}
				}
				
				
				#endregion
				
			}
			
		}
	}
}
