using System.Windows.Forms;
using Microsoft.VisualBasic;
using System;
using MicroFour.StrataFrame.Messaging;
using MicroFour.StrataFrame.Security;


namespace StrataFlix
{
	public partial class StrataFlixLogin : ILoginForm
	{
		
		#region " Constructors "
		
		/// <summary>
		/// Default Constructors
		/// </summary>
		/// <remarks></remarks>
		public StrataFlixLogin()
		{
			
			// This call is required by the Windows Form Designer.
			InitializeComponent();
			
			//-- Set the version information
			lblVersion.Text = System.Reflection.Assembly.GetExecutingAssembly().GetName().Version.ToString();
			
			#if DEBUG
			lblVersion.Text += " [Debug Build]";
			#endif
			
		}
		
		#endregion
		
		#region " Private Fields "
		
		private const string DomainValueName = "DomainValue";
		private static string _HeaderTitle = "User Authentication";
		private static string _HeaderDesc = "Please enter user credentials below.";
		
		#endregion
		
		#region " Events "
		
		/// <summary>
		/// Occurs when the end-user attempts to authenticate.
		/// </summary>
		/// <remarks></remarks>
		private EventHandler AttemptLoginEvent;
		public event EventHandler AttemptLogin
		{
			add
			{
				AttemptLoginEvent = (EventHandler) System.Delegate.Combine(AttemptLoginEvent, value);
			}
			remove
			{
				AttemptLoginEvent = (EventHandler) System.Delegate.Remove(AttemptLoginEvent, value);
			}
		}
		
		
		/// <summary>
		/// Raises the AttemptLogin event.
		/// </summary>
		/// <remarks></remarks>
		protected virtual void OnAttemptLogin()
		{
			if (AttemptLoginEvent != null)
				AttemptLoginEvent(this, EventArgs.Empty);
		}
		
		#endregion
		
		#region " Public Properties "
		
		/// <summary>
		/// Gets or sets a value that determines whether this form allows the application to be closed
		/// from within it.
		/// </summary>
		/// <value></value>
		/// <returns></returns>
		/// <remarks></remarks>
		public bool AllowAppExit
		{
			get
			{
				return this.cmdExit.Visible;
			}
			set
			{
				this.cmdExit.Visible = value;
			}
		}
		
		/// <summary>
		/// Gets or sets the domain name the end-user has entered.
		/// </summary>
		/// <value></value>
		/// <returns></returns>
		/// <remarks></remarks>
		public string DomainName
		{
			get
			{
				return this.cboDomain.Text;
			}
			set
			{
				this.cboDomain.Text = value;
			}
		}
		
		/// <summary>
		/// Gets or sets the gradient form header title.
		/// </summary>
		/// <value></value>
		/// <returns></returns>
		/// <remarks></remarks>
		public static string HeaderTitle
		{
			get
			{
				return _HeaderTitle;
			}
			set
			{
				_HeaderTitle = value;
			}
		}
		
		/// <summary>
		/// Gets or set the gradient form header description
		/// </summary>
		/// <value></value>
		/// <returns></returns>
		/// <remarks></remarks>
		public static string HeaderDesc
		{
			get
			{
				return _HeaderDesc;
			}
			set
			{
				_HeaderDesc = value;
			}
		}
		
		/// <summary>
		/// Gets or sets the password entered by the end-user.
		/// </summary>
		/// <value></value>
		/// <returns></returns>
		/// <remarks></remarks>
		public string Password
		{
			get
			{
				return this.txtPassword.Text;
			}
			set
			{
				this.txtPassword.Text = value;
			}
		}
		
		/// <summary>
		/// Gets or sets a value that indicates whether the domain combo box is shown.
		/// </summary>
		/// <value></value>
		/// <returns></returns>
		/// <remarks></remarks>
		public bool ShowDomainBox
		{
			get
			{
				return this.cboDomain.Visible;
			}
			set
			{
				this.lblDomain.Visible = value;
				this.cboDomain.Visible = value;
				//'-- Set the height of the form
				//If value Then
				//    Me.Height = 231
				//Else
				//    Me.Height = 205
				//End If
			}
		}
		
		/// <summary>
		/// Gets or sets the username entered by the end-user.
		/// </summary>
		/// <value></value>
		/// <returns></returns>
		/// <remarks></remarks>
		public string Username
		{
			get
			{
				return this.txtUsername.Text;
			}
			set
			{
				this.txtUsername.Text = value;
			}
		}
		
		#endregion
		
		#region " Protected Methods "
		
		/// <summary>
		/// Save off additional form settings prior to closing
		/// </summary>
		/// <param name="e"></param>
		/// <remarks></remarks>
		protected override void OnFormClosing(System.Windows.Forms.FormClosingEventArgs e)
		{
			//-- Save off the user name for next entry
			Registry.WriteValue("LastUser", txtUsername.Text);
			
			//-- Perform base logic
			base.OnFormClosing(e);
		}
		
		/// <summary>
		/// Overrides the OnLoad method of the base class.
		/// </summary>
		/// <param name="e"></param>
		/// <remarks></remarks>
		protected override void OnLoad(System.EventArgs e)
		{
			//-- Activate and show the form
			this.Activate();
			this.WindowState = FormWindowState.Normal;
			
			//-- Restore the last user login name
			this.txtUsername.Text = Registry.ReadString("LastUser", "Administrator");
			
			//-- Set the gradient form header
			//Me.GradientFormHeader1.Title = _HeaderTitle
			//Me.GradientFormHeader1.DetailText = _HeaderDesc
			
			//-- Base call
			base.OnLoad(e);
			
			//-- Set the focus to the password since the user should be filled out
			txtPassword.Focus();
		}
		
		#endregion
		
		#region " Public Methods "
		
		/// <summary>
		/// Sets the DialogResult of the form to DialogResult.OK.
		/// </summary>
		/// <remarks></remarks>
		public void SetDialogResultToOK()
		{
			this.DialogResult = System.Windows.Forms.DialogResult.OK;
		}
		
		/// <summary>
		/// Sets the delay seconds on the form.
		/// </summary>
		/// <param name="Value"></param>
		/// <remarks></remarks>
		public void SetDelaySeconds(int Value)
		{
			if (Value == 0)
			{
				this.grpLogin.Title = "Login Information";
			}
			else
			{
				this.grpLogin.Title = "Retry Delay: " + Value.ToString() + " seconds";
			}
		}
		
		/// <summary>
		/// Sets the domain names within the domain combo box to the given values.
		/// </summary>
		/// <param name="DomainNames"></param>
		/// <remarks></remarks>
		public void SetDomainNames(string[] DomainNames)
		{
			this.cboDomain.Items.Clear();
			this.cboDomain.Items.AddRange(DomainNames);
			
			//-- Set the text on the domain names to the old value
			this.cboDomain.Text = this.Registry.ReadString(DomainValueName, string.Empty);
		}
		
		/// <summary>
		/// Disable form and its controls
		/// </summary>
		/// <remarks></remarks>
		public void SetFormDisabled()
		{
			this.txtPassword.Enabled = false;
			this.txtUsername.Enabled = false;
			this.cboDomain.Enabled = false;
			this.cmdOk.Enabled = false;
			this.cmdCancel.Enabled = false;
		}
		
		/// <summary>
		/// Enable form and its controls
		/// </summary>
		/// <remarks></remarks>
		public void SetFormEnabled()
		{
			this.txtPassword.Enabled = true;
			this.txtUsername.Enabled = true;
			this.cboDomain.Enabled = true;
			this.cmdOk.Enabled = true;
			this.cmdCancel.Enabled = true;
		}
		
		/// <summary>
		/// Show the user account is deactivated.
		/// </summary>
		/// <remarks></remarks>
		public void ShowAccountDeactivated()
		{
			MessageForm.ShowMessageByKey("SFSM_UserDeactivated");
		}
		
		/// <summary>
		/// Shows an authentication failed message to the end-user.
		/// </summary>
		/// <remarks></remarks>
		public void ShowAuthFailedMessage()
		{
			MessageForm.ShowMessageByKey("SFSM_PasswordAuthenticationFailed");
		}
		
		/// <summary>
		/// Shows login attempts exceeded message.
		/// </summary>
		/// <remarks></remarks>
		public void ShowLoginAttemptsExceeded()
		{
			MessageForm.ShowMessageByKey("SFSM_LoginAttemptsExceeded");
		}
		
		/// <summary>
		/// Shows a message to the end-user informing them that they do not have the required permission
		/// necessary to login to the system.
		/// </summary>
		/// <remarks></remarks>
		public void ShowLoginPermissionDenied()
		{
			MessageForm.ShowMessageByKey("SFSM_LoginPermissionDenied");
		}
		
		#endregion
		
		#region " Event handlers "
		
		/// <summary>
		/// Handles the cmdOk.Click event.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		/// <remarks></remarks>
		private void cmdOk_Click(System.Object sender, System.EventArgs e)
		{
			//-- Save off the selected domain text
			if (this.cboDomain.Text.Length > 0)
			{
				this.Registry.WriteValue(DomainValueName, this.cboDomain.Text);
			}
			
			//-- Raise the attempt login event
			this.OnAttemptLogin();
		}
		
		/// <summary>
		/// Handles the cmdCancel.Click event.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		/// <remarks></remarks>
		private void cmdCancel_Click(System.Object sender, System.EventArgs e)
		{
			this.txtUsername.Text = string.Empty;
			this.txtPassword.Text = string.Empty;
			this.txtUsername.Focus();
		}
		
		/// <summary>
		/// Handles the cmdExit.Click event.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		/// <remarks></remarks>
		private void cmdExit_Click(System.Object sender, System.EventArgs e)
		{
			this.DialogResult = System.Windows.Forms.DialogResult.Cancel;
		}
		
		#endregion
		
	}
}
