﻿Imports System.ComponentModel
Imports MicroFour.StrataFrame.Security


''' <summary>
''' Base form used for project dependent forms. This provides 
''' a common set of functionality that all project editors
''' need. The following functionality is included in this base form.
''' 
''' <list>
''' <item>RAMS project header</item>
''' <item>RAMS icon</item>
''' <item>Handles updating header when non-current project is used</item>
''' </list>
''' </summary>
''' <remarks>
''' While all forms will need a toolstrip, putting a toolstripcontainer
''' on this from means we can't put ANY controls on derived forms.
''' </remarks>
Public Class RamsProjectBaseForm

#Region " Properties "

  Private _projectID As Integer

  ''' <summary>
  ''' The ProjectID of the current project
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' This is loaded when the form opens and can't be changed after that.
  ''' </remarks>
  Protected ReadOnly Property ProjectID() As Int32
    Get
      ProjectID = _projectID
    End Get
  End Property

  Private _fkField As String = String.Empty

  ''' <summary>
  ''' Define the name of the FK field. This will then allow for
  ''' the automatic setting of this field to the project ID. 
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' Leave it blank if the FK isn't set to the project ID.
  ''' </remarks>     
  <Category("RAMS: Project Editor") _
  , Description("Set the name of the field that is the FK field for the Project ID.")> _
  Public Property FKField() As String
    Get
      Return _fkField
    End Get
    Set(ByVal value As String)
      _fkField = value
    End Set
  End Property

  ''' <summary>
  ''' Define if the form should use a custom client name 
  ''' instead of the one defined based on the project. 
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' This is used with Standards and Templates because
  ''' their client is virutal, a place holder to simply 
  ''' contain the standards or templates.
  ''' 
  ''' The header will load the current client name as a 
  ''' matter of course when it is instantiated. However,
  ''' this flag will be set when the form loads and it
  ''' will then use the CustomClientName to load the 
  ''' client title into the header.
  ''' </remarks>     
  <Category("RAMS: Configure Header") _
 , Description("Indicate that a custom client name is used instead of the one associated with the project.")> _
  Public Property UseCustomClientName() As Boolean
    Get
      Return Me.ramsHeader.UseCustomClientName
    End Get
    Set(ByVal value As Boolean)
      Me.ramsHeader.UseCustomClientName = value
    End Set
  End Property

  ''' <summary>
  ''' Define a custom client name to use in the header.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>     
  <Category("RAMS: Configure Header") _
 , Description("The custom client name to use instead of the one associated with the project.")> _
  Public Property CustomClientName() As String
    Get
      Return Me.ramsHeader.CustomClientName
    End Get
    Set(ByVal value As String)
      Me.ramsHeader.CustomClientName = value
    End Set
  End Property

  ''' <summary>
  ''' Return the PK value of the current record. 
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' This assumes that a) there is a single PK field and
  ''' b) that it is an int.
  ''' </remarks>   
  <Browsable(False)> _
  Public ReadOnly Property CurrentID() As Integer
    Get
      If Me.BusinessObject Is Nothing Then Return 0
      If Me.BusinessObject.CurrentRowIndex < 0 Then Return 0
      Return DirectCast(Me.BusinessObject.Item(Me.BusinessObject.PrimaryKeyField), Integer)
    End Get
  End Property

#End Region

#Region " Methods "

  ''' <summary>
  ''' Set the project ID for the form.
  ''' </summary>
  ''' <remarks></remarks>
  Public Sub SetProjectID(ByVal projectID As Integer)
    ' Verify that we have a project
    'If projectID = Nothing OrElse projectID = 0 Then
    '  ' Error: must supply a project ID
    '  Throw New ArgumentException("The form require a valid project ID.")
    'End If

    ' Set the current project ID
    _projectID = projectID
  End Sub

#End Region

#Region " Event Handlers "

  ''' <summary>
  ''' Update the header with the corrected project if the project
  ''' in use isn't the current UserContext project.
  ''' </summary>
  ''' <param name="sender"></param>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Private Sub UpdateHeader(ByVal sender As Object, ByVal e As EventArgs) Handles Me.Load
    ' Load project name into description
    If Me.ProjectID <> 0 Then
      Me.ramsHeader.SetProject(Me.ProjectID)
    End If
  End Sub

  ''' <summary>
  ''' Handle setting the FK to the project ID if configured.
  ''' </summary>
  ''' <remarks></remarks>
  Protected Overrides Sub OnSetDefaultValues()
    If Me.FKField Is Nothing OrElse Me.FKField.Trim().Length() = 0 Then Return
    Me.BusinessObject.Item(Me.FKField) = Me.ProjectID
  End Sub

#End Region

End Class
