﻿Imports System.ComponentModel
Imports MicroFour.StrataFrame.Business
Imports MicroFour.StrataFrame.Security

''' <summary>
''' Provide the data entry area to manipulate/view requirement
''' remediation summary items. This form is designed to have 
''' a single summary item only.  
''' </summary>
''' <remarks></remarks>
Public Class RemediationSummaryPanel

#Region "   Events "

  ''' <summary>
  ''' Raise after the ProjectID property changes.
  ''' </summary>
  ''' <remarks></remarks>
  Public Event ProjectIDChanged As EventHandler

  ''' <summary>
  ''' Raise the ProjectIDChanged event.
  ''' </summary>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Protected Sub OnProjectIDChanged(ByVal e As EventArgs)
    RaiseEvent ProjectIDChanged(Me, e)
  End Sub

#End Region

#Region "   Properties "

  Private _securityKey As String = String.Empty

  ''' <summary>
  ''' Define the security key that controls security for this
  ''' panel.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>     
  <Category("RAMS: Remediation Summary Setup") _
  , Description("Enter the SecurityKey required for the user to access the data on the panel.") _
  , DefaultValue("")> _
  Public Property SecurityKey() As String
    Get
      Return _securityKey
    End Get
    Set(ByVal value As String)
      _securityKey = value
    End Set
  End Property

  Private _vendorID As Integer

  ''' <summary>
  ''' Define the vendor ID whose remediations are shown.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>     
  <Browsable(False)> _
  Public Property VendorID() As Integer
    Get
      Return _vendorID
    End Get
    Set(ByVal value As Integer)
      _vendorID = value
    End Set
  End Property

  Private _requirementKey As Integer

  ''' <summary>
  ''' Define the requirement key of the requirement whose 
  ''' remediations are shown.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>     
  <Browsable(False)> _
  Public Property RequirementKey() As Integer
    Get
      Return _requirementKey
    End Get
    Set(ByVal value As Integer)
      _requirementKey = value
    End Set
  End Property

  Private _projectID As Integer

  ''' <summary>
  ''' Define the project ID for the project the remediation is
  ''' apart of.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' This could be derived from the requirement or vendor (both are
  ''' project specific), however, any caller will likely know this 
  ''' information, thus eliminating the need for another call the db.
  ''' 
  ''' This also manages the OldProjectID, updating that to the former
  ''' value of the project ID. 
  ''' </remarks>     
  <Browsable(False)> _
  Public Property ProjectID() As Integer
    Get
      Return _projectID
    End Get
    Set(ByVal value As Integer)
      '-- Update the old project ID
      Me.OldProjectID = _projectID

      '-- Set the project ID
      _projectID = value

      '-- Raise the project ID changed event
      Me.OnProjectIDChanged(New EventArgs())
    End Set
  End Property

  Private _oldProjectID As Integer

  ''' <summary>
  ''' Define the "old" project ID or the last project
  ''' before the current one was set.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' Tracking this allows us to more effeciently manage
  ''' the score list, which is dependent on the project ID.
  ''' </remarks>     
  Private Property OldProjectID() As Integer
    Get
      Return _oldProjectID
    End Get
    Set(ByVal value As Integer)
      _oldProjectID = value
    End Set
  End Property

  Private _parentBaseForm As BaseForm

  ''' <summary>
  ''' Define a reference to the parent form cast as the BaseForm.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' This allows access to the SetControlState method.
  ''' </remarks>     
  Private Property ParentBaseForm() As BaseForm
    Get
      Return _parentBaseForm
    End Get
    Set(ByVal value As BaseForm)
      _parentBaseForm = value
    End Set
  End Property

  Private _showAuditCaption As String = "Show Auditing"

  ''' <summary>
  ''' Return the caption used when the toggle audit button will
  ''' show audit comments. 
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>     
  Private ReadOnly Property ShowAuditCaption() As String
    Get
      Return _showAuditCaption
    End Get
  End Property

  Private _hideAuditCaption As String = "Hide Auditing"

  ''' <summary>
  ''' Return the caption used when the toggle audit button wiil
  ''' hide audit comments. 
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>     
  Private ReadOnly Property HideAuditCaption() As String
    Get
      Return _hideAuditCaption
    End Get
  End Property

#End Region

#Region "   Methods "

  ''' <summary>
  ''' Load the remediation for the indicated project, requirement
  ''' and vendor.
  ''' </summary>
  ''' <param name="projectID"></param>
  ''' <param name="vendorID"></param>
  ''' <param name="requirementKey"></param>
  ''' <remarks></remarks>
  Public Sub LoadRemediation(ByVal projectID As Integer _
                           , ByVal vendorID As Integer _
                           , ByVal requirementKey As Integer)
    '-- Set the properties
    Me.ProjectID = projectID
    Me.VendorID = vendorID
    Me.RequirementKey = requirementKey

    '-- Load data
    Me.LoadData()
  End Sub

  ''' <summary>
  ''' Clear the form/BO of any remediation data.
  ''' </summary>
  ''' <remarks></remarks>
  Public Sub ClearRemediation()
    '-- Clear properties
    Me.ProjectID = 0
    Me.VendorID = 0
    Me.RequirementKey = 0

    '-- Clear BO
    'Me.ReqRemediationSummaryBO1.Clear()
  End Sub

  ''' <summary>
  ''' Handle setting up the form based on its current state.
  ''' </summary>
  ''' <param name="item"></param>
  ''' <param name="stateToCheck"></param>
  ''' <returns></returns>
  ''' <remarks>
  ''' This must be called by the parent form, in the base form
  ''' override for this function.
  ''' </remarks>
  Public Function GetControlStateViaForm(ByVal item As Object, ByVal stateToCheck As ControlState) As Boolean
    '-- Establish return variable. This defaults to true
    '   to handle the case of controls that are only affected by
    '   security, thus not handled here at all.
    Dim isSet As Boolean = True

    '-- Get the data needed for the rules. Note we don't
    '   need to do security here because this is called from 
    '   security handler.
    Dim isRecord As Boolean = True
    Dim eState As BusinessEditingState = BusinessEditingState.Idle
    Dim isIdle As Boolean = (eState = BusinessEditingState.Idle)
    Dim isLocked As Boolean = True
    Dim projectIsCurrent As Boolean = True
    Dim isDetailSelected As Boolean = isRecord AndAlso (Me.lvwDetails.SelectedItems.Count > 0)
    Dim requireCurrentProject As Boolean = False

    '-- This set of controls only checks for the Enabled state.
    If stateToCheck = ControlState.Enabled Then
      '-- tsbEdit enabled if the bo is idle and there is a record
      If item Is Me.tsbEdit Then
        isSet = isRecord AndAlso isIdle
        requireCurrentProject = True
      End If

      '-- tsbSave enabled if the BO is NOT idle
      If item Is Me.tsbSave Then
        isSet = Not isIdle
        requireCurrentProject = True
      End If

      '-- tsbUndo enabled if the BO is NOT idle
      If item Is Me.tsbUndo Then
        isSet = Not isIdle
        requireCurrentProject = True
      End If

      '-- tsbToggleAudit enabled if BO is NOT idle
      If item Is Me.tsbToggleAudit Then
        isSet = Not isIdle
        requireCurrentProject = True
      End If

      '-- tsbViewDetails is available if a detail is selected.
      If item Is Me.tsbViewDetails Then
        isSet = isDetailSelected
      End If

      '-- tsbAddDetails is available if there is record
      If item Is Me.tsbAddDetail Then
        isSet = isRecord
        requireCurrentProject = True
      End If

      '-- tsbRemoveDetails is available if a detail is selected
      If item Is Me.tsbRemoveDetail Then
        isSet = isDetailSelected
        requireCurrentProject = True
      End If

      '-- tsbRefresh is only available when not editing
      If item Is Me.tsbRefresh Then
        isSet = isIdle
      End If
    End If

    '-- If the control requires the current project, add that
    '   criterion also.
    If requireCurrentProject Then
      isSet = isSet AndAlso projectIsCurrent
    End If

    '-- Return if it is active
    Return isSet
  End Function

#End Region

#Region "   Event Handlers "

  ''' <summary>
  ''' Handle setting up the form based on its state.
  ''' </summary>
  ''' <remarks></remarks>
  Private Sub RemediationSummaryPanel_ParentFormLoading() Handles Me.ParentFormLoading
    '-- Make sure the audit spliter matches the show audit button
    If Me.tsbToggleAudit.Text = Me.ShowAuditCaption AndAlso Not Me.tscAuditing.Panel2Collapsed Then
      Me.tscAuditing.Panel2Collapsed = True
    End If

    '-- Validate that this form can be loaded on the form it is
    '   on, which must inherit from the BaseForm (FOX).
    If Not GetType(BaseForm).IsAssignableFrom(Me.ParentForm.GetType()) Then
      Throw New InvalidOperationException("This form cannot accept a RemediationSummaryPanel because does not inherit from BaseForm.")
    End If

    '-- Setup security
    Me.ParentBaseForm = DirectCast(Me.ParentForm, BaseForm)
    Me.ParentBaseForm.ViewSecurityKey = Me.SecurityKey

    '-- Configure control states
    Me.ParentBaseForm.SetControlStates()
  End Sub

  ''' <summary>
  ''' Set controls states based on list selection changing.
  ''' </summary>
  ''' <param name="sender"></param>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Private Sub lvwDetails_SelectedIndexChanged(ByVal sender As Object, ByVal e As System.EventArgs) Handles lvwDetails.SelectedIndexChanged
    Me.ParentBaseForm.SetControlStates()
  End Sub

  ''' <summary>
  ''' Set the project ID needed to load the scores for the
  ''' current project.
  ''' </summary>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Private Sub cboScore_ListPopulating(ByVal e As MicroFour.StrataFrame.UI.ListPopulatingEventArgs) Handles cboScore.ListPopulating
    e.Parameters(0).Value = Me.ProjectID
  End Sub

  ''' <summary>
  ''' Set the name of the enterprise list that holds the 
  ''' remediation actions.
  ''' </summary>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Private Sub cboRemediationAction_ListPopulating(ByVal e As MicroFour.StrataFrame.UI.ListPopulatingEventArgs) Handles cboRemediationAction.ListPopulating
    e.Parameters(0).Value = "RemediationAction"
  End Sub

  ''' <summary>
  ''' Set the parent remediation summary ID to load 
  ''' details for.
  ''' </summary>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Private Sub lvwDetails_ListPopulating(ByVal e As MicroFour.StrataFrame.UI.ListPopulatingEventArgs) Handles lvwDetails.ListPopulating
    '-- Set zero
    e.Parameters(0).Value = 0
  End Sub

  ''' <summary>
  ''' Edit the current remediation.
  ''' </summary>
  ''' <param name="sender"></param>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Private Sub tsbEdit_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles tsbEdit.Click
  End Sub

  ''' <summary>
  ''' Save any changes to the current remediation.
  ''' </summary>
  ''' <param name="sender"></param>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Private Sub tsbSave_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles tsbSave.Click
  End Sub

  ''' <summary>
  ''' Undo any changes to the current remediation.
  ''' </summary>
  ''' <param name="sender"></param>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Private Sub tsbUndo_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles tsbUndo.Click
  End Sub

  ''' <summary>
  ''' View the remediation details for the currently selected remediation
  ''' detail item.
  ''' </summary>
  ''' <param name="sender"></param>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Private Sub tsbViewDetails_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles tsbViewDetails.Click
    '-- If the parent form is open modally, then open the details modally
  End Sub

  ''' <summary>
  ''' Open the remediation detail form modally, allowing user to
  ''' add details, then update list.
  ''' </summary>
  ''' <param name="sender"></param>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Private Sub tsbAddDetail_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles tsbAddDetail.Click

  End Sub

  ''' <summary>
  ''' Remove the currently selected detail item after getting 
  ''' confirmation from the user.
  ''' </summary>
  ''' <param name="sender"></param>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Private Sub tsbRemoveDetail_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles tsbRemoveDetail.Click

  End Sub

  ''' <summary>
  ''' Update the data for the current requirement/vendor.
  ''' </summary>
  ''' <param name="sender"></param>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Private Sub tsbRefresh_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles tsbRefresh.Click
    Me.LoadData()
  End Sub

  ''' <summary>
  ''' Toggle between showing and hiding the audit comment area.
  ''' </summary>
  ''' <param name="sender"></param>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Private Sub tsbToggleAudit_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles tsbToggleAudit.Click
    '-- Determine if we are hiding or showing. This is determined
    '   by the text of the object.
    If Me.tsbToggleAudit.Text = Me.ShowAuditCaption Then
      '-- Show the audit section
      Me.tscAuditing.Panel2Collapsed = False

      '-- Configure for hiding
      Me.tsbToggleAudit.Text = Me.HideAuditCaption
    Else
      '-- Hide the audit section
      Me.tscAuditing.Panel2Collapsed = True

      '-- Configure for showing
      Me.tsbToggleAudit.Text = Me.ShowAuditCaption
    End If
  End Sub

  ''' <summary>
  ''' Close the parent form.
  ''' </summary>
  ''' <param name="sender"></param>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Private Sub tsbClose_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles tsbClose.Click
    Me.ParentForm.Close()
  End Sub

  ''' <summary>
  ''' Update the score list.
  ''' </summary>
  ''' <param name="sender"></param>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Private Sub RemediationSummaryPanel_ProjectIDChanged(ByVal sender As Object, ByVal e As System.EventArgs) Handles Me.ProjectIDChanged
    Me.LoadScores()
  End Sub

#End Region

#Region "   Private (helper) Methods "

  ''' <summary>
  ''' Load the score combo if the last project ID 
  ''' is different than the currnet project ID.
  ''' </summary>
  ''' <remarks></remarks>
  Private Sub LoadScores()
    If Me.OldProjectID <> Me.ProjectID Then
      Me.cboScore.Requery()
    End If
  End Sub

  ''' <summary>
  ''' Load data for the requirement and vendor specified 
  ''' by the class. If no remediation is found, then create
  ''' a remediation using the default score and action as 
  ''' defined by the BO.
  ''' </summary>
  ''' <remarks></remarks>
  Private Sub LoadData()
    '-- Validate that both the requirement key and the vendor
    '   have been specified. If both are not set, then return.
    '   This allows the caller to call this when either is changed
    '   but the other might not be set. 
    If Me.VendorID = 0 OrElse Me.RequirementKey = 0 Then Return

    '-- First attempt to load the BO

    '-- If no remediation was found, create a new one. This however
    '   is not straight forward. What if a reader is requesting this?
    '   They don't have field permssions to create the item.

    '-- Load details
    Me.lvwDetails.Requery()
  End Sub

#End Region

End Class
