﻿Imports System.ComponentModel
Imports System.Data
Imports System.Data.SqlClient
Imports MicroFour.StrataFrame.Data
Imports MicroFour.StrataFrame.Business


''' <summary>
''' Base form which provides a common set of functionality 
''' for any form that is data driven.
''' </summary>
''' <remarks>
''' This form provides the following features:
''' 
''' * Automatically set the current item's name to the 
''' form's caption
''' 
''' * Load a single item, based on the primary key of the 
''' form's BO.
''' 
''' * Load a set of records from a data table.
''' 
''' * Open the form for data entry (no records loaded, ready 
''' to add a new record).
''' 
''' * After loading a set of records, navigate to an item 
''' using the PK.
''' 
''' * After loading a set of records, navigate to a new 
''' item. 
''' 
''' * Provide properties to setup for form security. Manage 
''' securing the form on load, after save and on navigate.
''' 
''' * Provides an event that subscribers can handle when an 
''' item is saved.
''' 
''' * Provides a property that derived forms can use when the
''' form is being opened to return an item.
''' 
''' #DataLoadingOptions
''' There are several options that can be configured to 
''' control how data is loaded or what record is navigated 
''' to when the form is opened.  There are options to 
''' facilitate loading records into the primary BO and 
''' options to determine which record is shown to the user 
''' when the form is opened.
''' 
''' Loading Data
''' The form can help facilitate loading of data in a few 
''' ways. 
''' 
''' 1. Automatically load data based on a PK by setting the 
''' IdToLoad property.
''' 2. Load no data and automatically navigate to a new 
''' record by setting the DataEntry property to true.
''' 3. Load the data using an arbitrary fill method in the 
''' derived form and then navigate to a specific record 
''' based on the IdToNav property.
''' 4. Load the data using an arbitrary fill method in the 
''' derived form and then navigate to a new record.
''' 
''' #FactoryMethods
''' There is a set of shared factory methods that can be 
''' used to create forms that inherit from this base class.
''' These generic methods provide setup of a new form to 
''' use the features of the base form.
''' 
''' Derived classes can easily provide there own custom 
''' factories (say if they need to handle constructors)
''' by simply calling the appropriate constructor here, 
''' then updating the returned form before returning it.
''' 
''' Note that there are two functions for each type. The
''' 'Create' functions are used when you want that type 
''' returned. The 'Config' functions setup a passed in 
''' function.  
''' </remarks>
Public Class BaseForm
  Inherits MicroFour.StrataFrame.UI.Windows.Forms.StandardForm

#Region " Delegates and Events "

  ''' <summary>
  ''' Raised when the BO saves a record.
  ''' </summary>
  ''' <remarks></remarks>
  Public Event FormItemSaved As FormItemSavedEventHandler

  ''' <summary>
  ''' Raises the ProjectItemSaved event.
  ''' </summary>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Protected Overridable Sub OnFormItemSaved(ByVal e As FormItemSavedEventArgs)
    RaiseEvent FormItemSaved(Me, e)
  End Sub

#End Region

#Region "   Properties "

#Region "     Public Configuration Properties "

  'Private _controlSecurityProvider As FormUtils.ControlSecurityProvider = Nothing

  '''' <summary>
  '''' Define the control security provider used by the form.
  '''' </summary>
  '''' <value></value>
  '''' <returns></returns>
  '''' <remarks>
  '''' By setting this, this base form will manage the setting of
  '''' security for controls on the form.
  '''' </remarks>     
  '<Category("RAMS: Security Setup") _
  ', Description("If a ControlSecurityProvider is being used, select it. This will handle setting up and calling the ControlStateManager for the form.")> _
  'Public Property ControlSecurityProvider() As FormUtils.ControlSecurityProvider
  '  Get
  '    Return _controlSecurityProvider
  '  End Get
  '  Set(ByVal value As FormUtils.ControlSecurityProvider)
  '    _controlSecurityProvider = value
  '  End Set
  'End Property

  Private _businessObject As MicroFour.StrataFrame.Business.BusinessLayer = Nothing

  ''' <summary>
  ''' Define the business object that will be handling the
  ''' main data on the form. 
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' </remarks>     
  ''' 
  <Category("RAMS: Business Object Setup") _
  , Description("Set to main Business Object used by the form. This must be set to use the auto-caption feature and to use either the auto load or auto navigate features.")> _
  Public Property BusinessObject() As MicroFour.StrataFrame.Business.BusinessLayer
    Get
      Return _businessObject
    End Get
    Set(ByVal value As MicroFour.StrataFrame.Business.BusinessLayer)
      _businessObject = value
      AddHandler _businessObject.AfterSave, AddressOf Me.BusinessObject_AfterSave
      AddHandler _businessObject.Navigated, AddressOf Me.BusinessObject_Navigated
      AddHandler _businessObject.EditingStateChanged, AddressOf Me.BusinessObject_EditingStateChanged
      AddHandler _businessObject.SetDefaultValues, AddressOf Me.BusinessObject_SetDefaultValues
    End Set
  End Property

  Private _openForReturn As Boolean = False

  ''' <summary>
  ''' Set if the form is to be opened for return.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' This can be used by the specific instance to 
  ''' turn off navigation, limit the BO, etc.
  ''' </remarks>     
  <Category("RAMS: Close Method") _
  , Description("Indicate if the form is used to return a value to a caller. This can be used programmatically to setup the form to return data, control navigation, etc.")> _
  Public Property OpenForReturn() As Boolean
    Get
      Return _openForReturn
    End Get
    Set(ByVal value As Boolean)
      _openForReturn = value
    End Set
  End Property

  Private _itemNameProperty As String = String.Empty

  ''' <summary>
  ''' Define the name of the Business object property (field) 
  ''' that will return the name of the item. This is used to
  ''' add the name to the title. 
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>     
  <Category("RAMS: Auto-Caption Setup") _
  , Description("Set the name of the property/field that holds the name that will display in the form's caption.")> _
  Public Property ItemNameProperty() As String
    Get
      Return _itemNameProperty
    End Get
    Set(ByVal value As String)
      _itemNameProperty = value
    End Set
  End Property

  Private _titleTemplate As String = String.Empty

  ''' <summary>
  ''' Define the template used to build the form's caption.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' The caption will be built using the String.Format 
  ''' method. Place a '{0}' where you want the current item 
  ''' name to be placed.
  ''' </remarks>
  <Category("RAMS: Auto-Caption Setup") _
  , Description("The template used to add the current item's name to the form's caption. This should include a placeholder for the item name ('{0}'). If the placeholder doesn't exist, the template will be static.") _
  , DefaultValue("")> _
  Public Property TitleTemplate() As String
    Get
      Return _titleTemplate
    End Get
    Set(ByVal value As String)
      _titleTemplate = value
    End Set
  End Property

  Private _newItemName As String = String.Empty

  ''' <summary>
  ''' Define the text to use when a new item navigated to.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>     
  <Category("RAMS: Auto-Caption Setup") _
  , Description("The name of a new item. This is used with the TitleTemplate to build the form's caption when adding new items.") _
  , DefaultValue("")> _
  Public Property NewItemName() As String
    Get
      Return _newItemName
    End Get
    Set(ByVal value As String)
      _newItemName = value
    End Set
  End Property

  Private _noItemCaption As String = String.Empty

  ''' <summary>
  ''' Define the title when no item is selected.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' This is used when no item is selected. It does not get 
  ''' placed within the TitleTemplate, as the NewItemTitle 
  ''' does. This is used as-is as the title.
  ''' </remarks>
  <Category("RAMS: Auto-Caption Setup") _
  , Description("The caption to use if there is no current record. This would be the case if there are no records loaded and the BO is not on a new record.") _
  , DefaultValue("")> _
  Public Property NoItemCaption() As String
    Get
      Return _noItemCaption
    End Get
    Set(ByVal value As String)
      _noItemCaption = value
    End Set
  End Property

#End Region

#Region "     Designer Support Methods "

  '''' <summary>
  '''' Determine if the designer should serialize the ControlSecurityProvide
  '''' property.
  '''' </summary>
  '''' <returns></returns>
  '''' <remarks></remarks>
  'Private Function ShouldSerializeControlSecurityProvider() As Boolean
  '  Return _controlSecurityProvider IsNot Nothing
  'End Function

  '''' <summary>
  '''' Reset the ControlSecurityProvide property to its default value.
  '''' </summary>
  '''' <remarks></remarks>
  'Private Sub ResetControlSecurityProvider()
  '  _controlSecurityProvider = Nothing
  'End Sub

  ''' <summary>
  ''' Determine if the designer should serialize the BusinessObjec
  ''' property.
  ''' </summary>
  ''' <returns></returns>
  ''' <remarks></remarks>
  Private Function ShouldSerializeBusinessObject() As Boolean
    Return _businessObject IsNot Nothing
  End Function

  ''' <summary>
  ''' Reset the BusinessObjec property to its default value.
  ''' </summary>
  ''' <remarks></remarks>
  Private Sub ResetBusinessObject()
    _businessObject = Nothing
  End Sub

#End Region

#Region "     Internal Auto-load/Auto-nav Properties "

  Private _dataEntry As Boolean

  ''' <summary>
  ''' Define if the form is opened in data entry mode.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' This indicates that no data is loaded into the form and
  ''' the user is presented with the form ready to enter a 
  ''' new requirement.
  ''' </remarks> 
  Protected ReadOnly Property DataEntry() As Boolean
    Get
      Return _dataEntry
    End Get
  End Property

  Private _idToLoad As Integer

  ''' <summary>
  ''' Define the item ID used to initially load the form with
  ''' data.  
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' Once the form is loaded, the user can enter additional
  ''' requirements and therefore this won't be of use. Only
  ''' use this before or during the form's load event.
  ''' </remarks>    
  Protected ReadOnly Property IDToLoad() As Integer
    Get
      Return _idToLoad
    End Get
  End Property

  Private _data As Data.DataTable

  ''' <summary>
  ''' Define a data table with the data to load into the form.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' This is used to load a set of arbitrary data into the 
  ''' BO used by this form.
  ''' </remarks>
  Protected ReadOnly Property Data() As Data.DataTable
    Get
      Return _data
    End Get
  End Property

  Private _navOnLoadToNewRecord As Boolean = False

  ''' <summary>
  ''' Define if the form should navigate to a new record when 
  ''' the form loads. 
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' For a decription of the options available for data 
  ''' loading and/or navigation see code:#DataLoadingOptions 
  ''' </remarks>
  Protected ReadOnly Property NavOnLoadToNewRecord() As Boolean
    Get
      Return _navOnLoadToNewRecord
    End Get
  End Property

  Private _idToNav As Integer = 0

  ''' <summary>
  ''' The ID of the record to navigate to when the form 
  ''' loads.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' See NavOnLoadToNewRecord for more information.
  ''' </remarks>
  Protected ReadOnly Property IDToNav() As Integer
    Get
      Return _idToNav
    End Get
  End Property

#End Region

#Region "     Internal Class Capability Properties  "

  ''' <summary>
  ''' Return if the form is currently set to be able to 
  ''' automatically navigate to either a record or a new 
  ''' record after any data is loaded.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' The form can automatically navigate to a record if there
  ''' is a BusinessObject set and if either the IDToNav or the
  ''' NavOnLoadToNewRecord property is set.  
  ''' 
  ''' Not that this is independent of the loading of data and 
  ''' can be used in conjuntion with auto loading data. 
  ''' However, it really only makes sense if the Data property
  ''' is set and more than one item is loaded.
  ''' </remarks>
  Protected ReadOnly Property CanNavOnLoad() As Boolean
    Get
      '-- Get data needed to determine if the form can nav to
      '   an ID.
      Dim isBO As Boolean = (Me.BusinessObject IsNot Nothing)
      Dim isID As Boolean = (Me.IDToNav > 0)
      Dim isRecords As Boolean = isBO AndAlso (Me.BusinessObject.Count > 0)

      '-- In order for the form to auto nav, three conditions 
      '   need to be met an auto nav to a single record:
      '    1. business object must be set
      '    2. either an ID to nav to
      '    3. some records within the BO
      Dim canNavToID As Boolean = isBO AndAlso isID AndAlso isRecords

      '-- In order for the form to auto nave to a new record, 
      '   there are two conditions that must be met.
      '    1. business object must be set
      '    2. dataentry property is set
      Dim canNavToNew As Boolean = isBO AndAlso Me.NavOnLoadToNewRecord

      Return (canNavToID OrElse canNavToNew)
    End Get
  End Property

  ''' <summary>
  ''' Return if the form is currently set to be able to 
  ''' automatically load data into the BO. 
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' The form can automatically load data into the BO if the 
  ''' BusinessObject property is set and either the DataEntry 
  ''' property is true or the IdToLoad property is > 0 or the 
  ''' Data property is set.
  ''' </remarks>
  Protected ReadOnly Property CanAutoLoadBO() As Boolean
    Get
      Return (Me.BusinessObject IsNot Nothing AndAlso _
                    (Me.DataEntry OrElse Me.IDToLoad > 0 OrElse Me.Data IsNot Nothing))
    End Get
  End Property

  ''' <summary>
  ''' Return if the form is configured to be able to load the 
  ''' form's caption 
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' The caption can be loaded automatically if the 
  ''' BusinessObject is set and also the ItemNameProperty.
  ''' </remarks>
  Protected ReadOnly Property CanSetFormCaption() As Boolean
    Get
      '-- Can be set if there is a BO and column to use for the name is set.
      Return Me.BusinessObject IsNot Nothing AndAlso Not String.IsNullOrEmpty(Me.ItemNameProperty)
    End Get
  End Property

#End Region

#Region "     Internal Control State Properties "

  'Private _stateManager As FormUtils.ControlStateManager

  '''' <summary>
  '''' Return the ControlStateManager used by the form.
  '''' </summary>
  '''' <value></value>
  '''' <returns></returns>
  '''' <remarks></remarks>     
  'Private ReadOnly Property StateManager() As FormUtils.ControlStateManager
  '  Get
  '    '-- Instantiate if it isn't already
  '    If _stateManager Is Nothing Then
  '      _stateManager = New FormUtils.ControlStateManager()
  '    End If
  '    Return _stateManager
  '  End Get
  'End Property

#End Region

#End Region

#Region "   Methods "

#Region "     Configuration Setup Methods for Auto-load and Auto-nav "

  ''' <summary>
  ''' Set the form to do data entry when loaded.
  ''' </summary>
  ''' <param name="value"></param>
  ''' <remarks></remarks>
  Public Sub SetDataEntry(ByVal value As Boolean)
    _dataEntry = value
  End Sub

  ''' <summary>
  ''' Set the form to load the indicated ID when loaded.
  ''' </summary>
  ''' <param name="value"></param>
  ''' <remarks></remarks>
  Public Sub SetIdToLoad(ByVal value As Integer)
    _idToLoad = value
  End Sub

  ''' <summary>
  ''' Set the data that the form should load.
  ''' </summary>
  ''' <param name="value"></param>
  ''' <remarks></remarks>
  Public Sub SetDataToLoad(ByVal value As Data.DataTable)
    _data = value
  End Sub

  ''' <summary>
  ''' Set the form to force manual loading of data, 
  ''' even if it is setup such that data could be
  ''' auto loaded.
  ''' </summary>
  ''' <remarks></remarks>
  Public Sub SetForceManualLoad()
    Me._data = Nothing
    Me._idToLoad = 0
    Me._dataEntry = False
  End Sub

  ''' <summary>
  ''' Set the ID to navigate to after loading data.
  ''' </summary>
  ''' <param name="value"></param>
  ''' <remarks></remarks>
  Public Sub SetIdToNav(ByVal value As Integer)
    _idToNav = value
  End Sub

  ''' <summary>
  ''' Set that the form should navigate to a new record after 
  ''' loading data.
  ''' </summary>
  ''' <param name="value"></param>
  ''' <remarks></remarks>
  Public Sub SetNavOnLoadToNew(ByVal value As Boolean)
    _navOnLoadToNewRecord = value
  End Sub

#End Region

#Region "     Reload Methods "

  ''' <summary>
  ''' Reload the data, based on the auto-load settings for the
  ''' BO, then navigate to the indicated ID.
  ''' </summary>
  ''' <param name="currID"></param>
  ''' <returns></returns>
  ''' <remarks>
  ''' </remarks>
  Public Function ReloadData(ByVal currID As Integer) As Boolean
    '-- Establish return
    Dim wasAutoLoaded As Boolean = False

    '-- If the BO isn't setup for auto-loading return false
    If Not Me.CanAutoLoadBO Then Return wasAutoLoaded

    '-- Redo the auto load.
    Me.AutoLoadData()
    wasAutoLoaded = True

    '-- Setup bo for auto nav and then do the nav
    If currID > 0 Then
      Me.SetIdToNav(currID)
      If Me.CanNavOnLoad() Then
        Me.AutoNav()
      End If
    End If

    '-- Return if auto-loaded 
    Return wasAutoLoaded
  End Function

  ''' <summary>
  ''' Load a single record into the BO and navigate to that 
  ''' newly updated/loaded record.
  ''' </summary>
  ''' <param name="id"></param>
  ''' <remarks></remarks>
  Public Sub LoadSingleRecord(ByVal id As Integer)
    Me.LoadSingleRecord(id, True)
  End Sub

  ''' <summary>
  ''' Load a single record into the BO. The loaded record can
  ''' be navigated to if indicated. This can be used to either 
  ''' load new records into a BO or to update existing 
  ''' records.
  ''' </summary>
  ''' <param name="id">ID of record to load into the BO.
  ''' </param>
  ''' <param name="navToID">indicate if the BO should navigate
  ''' to the newly added record
  ''' </param>
  ''' <remarks>
  ''' This uses the Table and PrimaryKey properties of the BO 
  ''' to build a sql statement to do the load. Because of this
  ''' this shouldn't be used if there are other required extra
  ''' columns that need to be loaded. 
  ''' </remarks>
  Public Sub LoadSingleRecord(ByVal id As Integer _
                            , ByVal navToID As Boolean)
    '-- Build SQL to load a single record
    Dim sqlBuilder As New System.Text.StringBuilder(255)
    With sqlBuilder
      .AppendLine("Select *")
      .AppendFormat("From {0}", Me.BusinessObject.TableName).Append(ControlChars.NewLine)
      .AppendFormat("Where {0} = @pk", Me.BusinessObject.PrimaryKeyField).Append(ControlChars.NewLine)
    End With

    '-- Load the single record
    Using cmd As New SqlCommand()
      cmd.CommandText = sqlBuilder.ToString()

      cmd.Parameters.Add("@pk", SqlDbType.Int)
      cmd.Parameters("@pk").Value = id

      '-- Update the BO to include the new record. Note that if 
      '   the record is already in the BO, it is reloaded by the 
      '   new data.
      Me.BusinessObject.AppendDataTable(cmd, AppendDataTableOptions.ReplaceExistingDuplicates)
    End Using

    '-- Now navigate to this new record if indicated.
    If navToID Then
      Me.BusinessObject.NavigateToPrimaryKey(id)
    End If
  End Sub

#End Region

#Region "     Control State Management "

  '''' <summary>
  '''' Setup control state management for the form.
  '''' </summary>
  '''' <param name="provider">a ControlSecurityProvider</param>
  '''' <param name="formCallback">delegate called to manage form level logic</param>
  '''' <remarks>
  '''' This is done automatically in the OnLoad method which 
  '''' works fine as long as the ControlSecurityProvider is 
  '''' set. However, there are time when this must be setup 
  '''' manually. In those cases this can be called manually
  '''' to configure control state management.
  '''' </remarks>
  'Public Sub ConfigControlStateManagement(ByVal provider As FormUtils.ControlSecurityProvider _
  '                                      , ByVal formCallback As ControlStateCallback)
  '  '-- If there is no ControlSecurityProvider, return
  '  If provider Is Nothing Then Return

  '  '-- Configure the ControlStateManager
  '  With Me.StateManager
  '    .CheckList = provider.GetCheckList
  '    .FormLogicCallback = formCallback
  '  End With
  'End Sub

  ''' <summary>
  ''' Set the states (visible and enabled) of controls that
  ''' are manage by the ControlStateManager (defined by the
  ''' ControlSecurityProvider).  
  ''' </summary>
  ''' <remarks></remarks>
  Public Sub SetControlStates()
    '-- use the control state manager
    'Me.StateManager.ProcessStates()
  End Sub

#End Region

#End Region

#Region "   Shared Factory Methods "

  ''' <summary>
  ''' Create and return a form setup for data entry.
  ''' </summary>
  ''' <typeparam name="T">type derived from BaseForm</typeparam>
  ''' <returns>new form of type T configured for data entry</returns>
  ''' <remarks></remarks>
  Public Shared Function CreateDataEntryForm(Of T As {New, BaseForm})() As T
    Dim frm As New T
    BaseForm.ConfigDataEntryForm(frm)
    Return frm
  End Function

  ''' <summary>
  ''' Configure the provided form derived from the BaseForm for 
  ''' data entry.
  ''' </summary>
  ''' <typeparam name="T">type derived from BaseForm</typeparam>
  ''' <param name="form">form to configure</param>
  ''' <remarks>
  ''' Note that this not only turns on data entry but also turns 
  ''' off any other auto-loading.
  ''' </remarks>
  Public Shared Sub ConfigDataEntryForm(Of T As BaseForm)(ByRef form As T)
    form.SetDataEntry(True)
    form.SetIdToLoad(0)
    form.SetDataToLoad(Nothing)
  End Sub

  ''' <summary>
  ''' Create and return a form that loads the indicated record.
  ''' </summary>
  ''' <typeparam name="T">type derived from BaseForm</typeparam>
  ''' <param name="id">id of record to load</param>
  ''' <returns>new form of type T configured for single record auto-loading</returns>
  ''' <remarks></remarks>
  Public Shared Function CreateSingleRecordForm(Of T As {New, BaseForm})(ByVal id As Integer) As T
    Dim frm As New T
    BaseForm.ConfigSingleRecordForm(frm, id)
    Return frm
  End Function

  ''' <summary>
  ''' Configure the provided form as a single record form (i.e.
  ''' a form whose BO has loaded a single record).
  ''' </summary>
  ''' <typeparam name="T">type derived from BaseForm</typeparam>
  ''' <param name="form">form to configure</param>
  ''' <param name="id">id of record to load</param>
  ''' <remarks>
  ''' Note that this not only turns single record auto-loading
  ''' but also turns off any other auto-loading.
  ''' </remarks>
  Public Shared Sub ConfigSingleRecordForm(Of T As BaseForm)(ByRef form As T, ByVal id As Integer)
    form.SetIdToLoad(id)
    form.SetDataEntry(False)
    form.SetDataToLoad(Nothing)
  End Sub

  ''' <summary>
  ''' Create and return a form that loads the records in the 
  ''' provided data table.
  ''' </summary>
  ''' <typeparam name="T">type derived from BaseForm</typeparam>
  ''' <param name="data">DataTable with data to load</param>
  ''' <returns>new form of type T configured for multiple record auto-loading</returns>
  ''' <remarks></remarks>
  Public Shared Function CreateDataLoadedForm(Of T As {New, BaseForm})(ByVal data As DataTable) As T
    Dim frm As New T
    BaseForm.ConfigDataLoadedForm(frm, data)
    Return frm
  End Function

  ''' <summary>
  ''' Configure the provided form to be auto-loaded form 
  ''' the provided data table.
  ''' </summary>
  ''' <typeparam name="T">type derived from BaseForm</typeparam>
  ''' <param name="form">form to be configured</param>
  ''' <param name="data">DataTable with data to load</param>
  ''' <remarks></remarks>
  Public Shared Sub ConfigDataLoadedForm(Of T As BaseForm)(ByVal form As T, ByVal data As DataTable)
    form.SetDataToLoad(data)
    form.SetIdToLoad(0)
    form.SetDataEntry(False)
  End Sub

  ''' <summary>
  ''' Create and return a form that navigates to a new record after 
  ''' it is loaded with data.
  ''' </summary>
  ''' <typeparam name="T">type derived from BaseForm</typeparam>
  ''' <returns></returns>
  ''' <remarks></remarks>
  Public Shared Function CreateNavToNewForm(Of T As {New, BaseForm})() As T
    Dim frm As New T
    BaseForm.ConfigNavToNewForm(frm)
    Return frm
  End Function

  ''' <summary>
  ''' Configure the provided form to navigate to a new record after any 
  ''' data is loaded.
  ''' </summary>
  ''' <typeparam name="T">type derived from BaseForm</typeparam>
  ''' <param name="form">form to be configured</param>
  ''' <remarks></remarks>
  Public Shared Sub ConfigNavToNewForm(Of T As BaseForm)(ByRef form As T)
    form.SetNavOnLoadToNew(True)
    form.SetIdToNav(0)
  End Sub

  ''' <summary>
  ''' Create and return a form that navigates to the indicated record
  ''' after the form has loaded any data.
  ''' </summary>
  ''' <typeparam name="T">type derived from BaseForm</typeparam>
  ''' <param name="id">id of record to load</param>
  ''' <returns>form of type T configured to auto-nav to the indicated record.</returns>
  ''' <remarks></remarks>
  Public Shared Function CreateNavToRecordForm(Of T As {New, BaseForm})(ByVal id As Integer) As T
    Dim frm As New T
    BaseForm.ConfigNavToRecordForm(frm, id)
    Return frm
  End Function

  ''' <summary>
  ''' Configure the provided form to auto-navigate to the indicated
  ''' record after the form is loaded.
  ''' </summary>
  ''' <typeparam name="T">type derived from BaseForm</typeparam>
  ''' <param name="form">Form to configure</param>
  ''' <param name="id">ID of record to navigate to</param>
  ''' <remarks></remarks>
  Public Shared Sub ConfigNavToRecordForm(Of T As BaseForm)(ByRef form As T, ByVal id As Integer)
    form.SetIdToNav(id)
    form.SetNavOnLoadToNew(False)
  End Sub

  ''' <summary>
  ''' Create and configure a form that loads the data from the 
  ''' provided DataTable, then auto-navigates to the indicated 
  ''' record.
  ''' </summary>
  ''' <typeparam name="T">type derived from BaseForm</typeparam>
  ''' <param name="data">DataTable with data to load</param>
  ''' <param name="id">Id of record to auto-navigate to</param>
  ''' <returns></returns>
  ''' <remarks></remarks>
  Public Shared Function CreateAutoLoadAndNavForm(Of T As {New, BaseForm})(ByVal data As DataTable, ByVal id As Integer) As T
    Dim frm As New T
    BaseForm.ConfigDataLoadedForm(frm, data)
    BaseForm.ConfigNavToRecordForm(frm, id)
    Return frm
  End Function

#End Region

#Region "   Form Specific Overridable Methods "

  ''' <summary>
  ''' Dummy method that is a placeholder for derived forms to
  ''' override in order to implement control state logic.
  ''' </summary>
  ''' <param name="item"></param>
  ''' <param name="stateToCheck"></param>
  ''' <returns></returns>
  ''' <remarks>
  ''' This allows this base class to call this function, but then 
  ''' let derived forms to override it and provide logic.
  ''' 
  ''' This is used to provide form level logic about if a control
  ''' should be visible or enabled.
  ''' </remarks>
  Protected Overridable Function GetControlStateViaForm(ByVal item As Object _
                                                      , ByVal stateToCheck As ControlState) As Boolean
    Return True
  End Function

#End Region

#Region "   Event Handlers "

  ''' <summary>
  ''' Update the form's caption after an item is saved.
  ''' </summary>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Public Sub BusinessObject_AfterSave(ByVal e As AfterSaveUndoEventArgs)
    '-- Only update the caption if the form is setup to
    '   handle auto-caption updates
    If Me.CanSetFormCaption Then
      Me.SetFormCaption()
    End If

    '-- Get the ID of the current ID
    Dim id As Integer
    If Me.BusinessObject.CurrentRowIndex >= 0 Then
      id = CInt(Me.BusinessObject.CurrentRow.Item(Me.BusinessObject.PrimaryKeyField))
    End If

    '-- Raise event that item was just saved
    Me.OnFormItemSaved(New FormItemSavedEventArgs(id, e.RowsAffected))

    '-- Manage control state
    Me.SetControlStates()
  End Sub

  ''' <summary>
  ''' Update the form's caption after navigation.
  ''' </summary>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Public Sub BusinessObject_Navigated(ByVal e As NavigatedEventArgs)
    '-- Only update the caption if the form is setup to
    '   handle auto-caption updates
    If Me.CanSetFormCaption Then
      Me.SetFormCaption()
    End If

    '-- Manage control state
    Me.SetControlStates()
  End Sub

  ''' <summary>
  ''' Manage control state during editing state changes.
  ''' </summary>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Public Sub BusinessObject_EditingStateChanged(ByVal e As EditingStateChangedEventArgs)
    Me.SetControlStates()
  End Sub

  ''' <summary>
  ''' Handle the SetDefaultValues event, so derived 
  ''' classes can react to this event. 
  ''' </summary>
  ''' <remarks>
  ''' Derived classes simply override the OnSetDefaultValues
  ''' method and do what ever they need. Note there is not need
  ''' (at this point) to call MyBase.OnSetDefaultValues, as it
  ''' does nothing.
  ''' </remarks>
  Public Sub BusinessObject_SetDefaultValues()
    Me.OnSetDefaultValues()
  End Sub

  ''' <summary>
  ''' Provides an overridable method so derived classes
  ''' can react to the SetDefaultValues event.
  ''' </summary>
  ''' <remarks></remarks>
  Protected Overridable Sub OnSetDefaultValues()
  End Sub

  ''' <summary>
  ''' Override the OnLoad method so we can automatically load 
  ''' the BusinessObject with data if needed and also handle 
  ''' any auto navigation.
  ''' </summary>
  ''' <param name="e"></param>
  ''' <remarks>
  ''' This does any auto-loading of data BEFORE the base class
  ''' load event is called, and navigation AFTER the base 
  ''' class load if called.
  ''' 
  ''' Thus, when reacting to the Load event in a derived form,
  ''' if any data is auto loaded, it is available within that 
  ''' event.
  ''' 
  ''' If the derived form will be navigating, the data should 
  ''' be loaded either in the form's Load event handler or in 
  ''' the ParentFormLoading handler for the BO.
  ''' </remarks>
  Protected Overrides Sub OnLoad(ByVal e As System.EventArgs)
    '-- If the form is in designer mode, just call the base on load
    If Me.DesignMode Then
      MyBase.OnLoad(e)
      Return
    End If

    '-- Setup Control state management
    'Me.ConfigControlStateManagement(Me.ControlSecurityProvider _
    '                              , AddressOf Me.GetControlStateViaForm)

    '-- If the form can auto load the BO, do it now
    If Me.CanAutoLoadBO Then
      Me.AutoLoadData()
    End If

    '-- Call the base class OnLoad. This will handle loading 
    '   any data that is loaded during the ParentFormLoading
    '   event and it will also resize/locate the form. It 
    '   also raises the Load event, so data can be loaded by
    '   handlers of that event also.
    MyBase.OnLoad(e)

    '-- If the form can auto nav, do it now
    If Me.CanNavOnLoad Then
      Me.AutoNav()
    End If

    '-- If there is no data in the BO, set the caption
    '   as it won't have navigated.
    If Me.BusinessObject IsNot Nothing AndAlso Me.BusinessObject.Count = 0 Then
      Me.SetFormCaption()
    End If

    '-- Secure controls...data is likely loaded
    Me.SetControlStates()
  End Sub

#End Region

#Region "   Base Class Override Methods "

  ''' <summary>
  ''' Handle cleaning up event handlers set by this base 
  ''' class.
  ''' </summary>
  ''' <param name="disposing"></param>
  ''' <remarks></remarks>
  <System.Diagnostics.DebuggerNonUserCode()> _
  Protected Overrides Sub Dispose(ByVal disposing As Boolean)
    '-- Remove the event handlers
    If disposing Then
      If _businessObject IsNot Nothing Then
        RemoveHandler _businessObject.AfterSave, AddressOf Me.BusinessObject_AfterSave
        RemoveHandler _businessObject.Navigated, AddressOf Me.BusinessObject_Navigated
        RemoveHandler _businessObject.EditingStateChanged, AddressOf Me.BusinessObject_EditingStateChanged
        RemoveHandler _businessObject.SetDefaultValues, AddressOf Me.BusinessObject_SetDefaultValues
      End If
    End If

    '-- Dispose of any components on the form.
    If disposing AndAlso components IsNot Nothing Then
      components.Dispose()
    End If

    '-- Call base class disposing
    MyBase.Dispose(disposing)
  End Sub

#End Region

#Region "   Private (helper) Methods "

  ''' <summary>
  ''' Set the form's caption based on the current record.
  ''' </summary>
  ''' <remarks></remarks>
  Private Sub SetFormCaption()
    '-- Only update the caption if the form is setup to
    '   handle auto-caption updates
    If Not Me.CanSetFormCaption Then Return

    '-- Get the name of the current item, or if it is
    '   a new item or if no item is selected.
    If Me.BusinessObject.Count = 0 Then
      '-- No current item
      Me.Text = String.Format(Me.TitleTemplate, Me.NoItemCaption)
    Else
      '-- See if the item is new
      If Me.BusinessObject.CurrentRow.RowState = DataRowState.Added Then
        '-- It is a new item
        Me.Text = String.Format(Me.TitleTemplate, Me.NewItemName)
      Else
        '-- Not a new item, use the items name
        Dim itemName As String = CType(Me.BusinessObject.Item(Me.ItemNameProperty), String)
        Me.Text = String.Format(Me.TitleTemplate, itemName)
      End If
    End If
  End Sub

  ''' <summary>
  ''' Auto load data if the form is setup to be auto loaded.
  ''' </summary>
  ''' <remarks>
  ''' There are three ways the data can auto-load the BO. The 
  ''' first is to load the BO from a datatable. The next is to
  ''' load a single record based on an ID. The final is to 
  ''' load a new record.  If the form is setup to load more 
  ''' than one way, the above priority will be used.
  ''' </remarks>
  Private Sub AutoLoadData()
    '-- Determine if the form can be auto loaded at all
    If Not Me.CanAutoLoadBO Then Return

    '-- If there is a datatable, use that data
    If Me.Data IsNot Nothing Then
      '-- Copy the data from the datatable into the BO
      Me.BusinessObject.CopyDataFrom(Me.Data, BusinessCloneDataType.ClearAndFillFromCompleteTable)
    ElseIf Me.IDToLoad > 0 Then
      '-- Load this ID, which must be the PK
      Me.BusinessObject.FillByPrimaryKey(Me.IDToLoad)
    Else
      '-- Put form into data entry mode...navigate to a new record
      '   without loading any data.
      Me.BusinessObject.Add()
    End If
  End Sub

  ''' <summary>
  ''' Auto navigate to either the IDToNav or to a new record, 
  ''' based on the configuration of the form.
  ''' </summary>
  ''' <remarks>
  ''' It is possible to have an ID that doesn't exist in the 
  ''' loaded data of the BO.
  ''' </remarks>
  Private Sub AutoNav()
    '-- Determine if the form is configured to auto nav
    If Not Me.CanNavOnLoad Then Return

    '-- If there is an ID to load and records in the bo
    '   then navigate to BO to that record
    If Me.BusinessObject.Count > 0 AndAlso Me.IDToNav > 0 Then
      '-- Navigate to the indicated ID (which must be an ID of the PK column)
      Me.BusinessObject.NavigateToPrimaryKey(Me.IDToNav)
    Else
      '-- Must be data entry
      Me.BusinessObject.Add()
    End If
  End Sub

#End Region

End Class
