Imports System.ComponentModel

Public Class RamsProjectHeader

#Region " Constructors "

  ''' <summary>
  ''' Default constructor which loads the project name and client
  ''' name from the current UserContext.
  ''' </summary>
  ''' <remarks></remarks>
  Public Sub New()
    Me.InitializeComponent()

    ' Skip loading at design time
    If Me.DesignMode Then Return

    ' Setup known info from UserContext
    Me.SetupTitles(0)
  End Sub

#End Region

#Region " Customization Properties "

  ''' <summary>
  ''' The title that is displayed in the header.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>
  <Category("RAMS: Configure Header") _
  , Description("The title that is displayed in the header.")> _
  Public Property Title() As String
    Get
      Return Me.ramsHeader.Title
    End Get
    Set(ByVal value As String)
      Me.ramsHeader.Title = value
    End Set
  End Property

  ''' <summary>
  ''' The detail text that is displayed in the header.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>
  <Category("RAMS: Configure Header") _
  , Description("The detail text that is displayed in the header.")> _
  Public Property DetailText() As String
    Get
      Return Me.ramsHeader.DetailText
    End Get
    Set(ByVal value As String)
      Me.ramsHeader.DetailText = value
    End Set
  End Property

  Private _detailsFormatString As String = String.Empty

  ''' <summary>
  ''' Define the format string that will be used to generate the
  ''' details text in the header. This should include a single
  ''' replacable parameter, {0}, which will be replaced by the
  ''' formatted name of the project. 
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' While you can set the full details text, this allows you
  ''' to set the template at design time and have the details
  ''' text built at runtime. Use this if the you want the project
  ''' name in the details text.
  ''' </remarks>     
  <Category("RAMS: Configure Header") _
  , Description("The a String.Format template that is used to insert the project name into the details text.") _
  , DefaultValue("")> _
  Public Property DetailsFormatString() As String
    Get
      Return _detailsFormatString
    End Get
    Set(ByVal value As String)
      _detailsFormatString = value
    End Set
  End Property

  Private _useCustomClientName As Boolean

  ''' <summary>
  ''' Define if the form should use a custom client name 
  ''' instead of the one defined based on the project. 
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' This is used with Standards and Templates because
  ''' their client is virutal, a place holder to simply 
  ''' contain the standards or templates.
  ''' 
  ''' The header will load the current client name as a 
  ''' matter of course when it is instantiated. However,
  ''' this flag will be set when the form loads and it
  ''' will then use the CustomClientName to load the 
  ''' client title into the header.
  ''' </remarks>     
  <Category("RAMS: Configure Header") _
 , Description("Indicate that a custom client name is used instead of the one associated with the project.")> _
Public Property UseCustomClientName() As Boolean
    Get
      Return _useCustomClientName
    End Get
    Set(ByVal value As Boolean)
      _useCustomClientName = value
    End Set
  End Property

  Private _customClientName As String

  ''' <summary>
  ''' Define a custom client name to use in the header.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>     
  <Category("RAMS: Configure Header") _
  , Description("The custom client name to use instead of the one associated with the project.")> _
  Public Property CustomClientName() As String
    Get
      Return _customClientName
    End Get
    Set(ByVal value As String)
      _customClientName = value
    End Set
  End Property

  ''' <summary>
  ''' The client name that is being displayed.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' To set this, use SetClientName.
  ''' </remarks>
  <Browsable(False)> _
  Public ReadOnly Property ClientName() As String
    Get
      Return Me.lblClientNameTitle.Text
    End Get
  End Property

  ''' <summary>
  ''' The project name being displayed in the header.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' To set this, use SetProjectName.
  ''' </remarks>
  <Browsable(False)> _
  Public ReadOnly Property ProjectName() As String
    Get
      Return Me.lblProjectNameTitle.Text
    End Get
  End Property

#End Region

#Region " Methods "

  ''' <summary>
  ''' This allows the caller to set the project ID, which will
  ''' then load the project name and the client name.
  ''' </summary>
  ''' <param name="projectID"></param>
  ''' <remarks></remarks>
  Public Sub SetProject(ByVal projectID As System.Int32)
    Me.SetupTitles(projectID)
  End Sub

  ''' <summary>
  ''' Set the project name on the header.
  ''' </summary>
  ''' <param name="projectName"></param>
  ''' <remarks></remarks>
  Public Sub SetProjectName(ByVal projectName As String)
    Me.lblProjectNameTitle.Text = projectName
    Me.Text = projectName
  End Sub

  ''' <summary>
  ''' Set the client name on the header.
  ''' </summary>
  ''' <param name="clientName"></param>
  ''' <remarks></remarks>
  Public Sub SetClientName(ByVal clientName As String)
    Me.lblClientNameTitle.Text = clientName
  End Sub

  ''' <summary>
  ''' Set the labels for client and project in the header,
  ''' leaving the widht alone.
  ''' </summary>
  ''' <param name="clientLabel"></param>
  ''' <param name="projectLabel"></param>
  ''' <remarks></remarks>
  Public Sub SetLabels(ByVal clientLabel As String, ByVal projectLabel As String)
    Me.SetLabels(clientLabel, 0, projectLabel, 0)
  End Sub

  ''' <summary>
  ''' Set the labels for client and project in the header and
  ''' also specify the width of the labels.
  ''' </summary>
  ''' <param name="clientLabel">text for client label</param>
  ''' <param name="clientWidth">width of the client label, 0 means no change</param>
  ''' <param name="projectLabel">text for project label</param>
  ''' <param name="projectWidth">widht of project label, 0 means no change</param>
  ''' <remarks></remarks>
  Public Sub SetLabels(ByVal clientLabel As String _
                      , ByVal clientWidth As Integer _
                      , ByVal projectLabel As String _
                      , ByVal projectWidth As Integer)
    '-- Set the labels
    Me.lblClient.Text = clientLabel
    Me.lblProject.Text = projectLabel

    '-- If indicated, set the widths
    If clientWidth > 0 Then
      Me.tlpInfo.ColumnStyles(0).SizeType = Windows.Forms.SizeType.Absolute
      Me.tlpInfo.ColumnStyles(0).Width = clientWidth
      Me.lblClient.Width = clientWidth
    End If
    If projectWidth > 0 Then
      Me.tlpInfo.ColumnStyles(2).SizeType = Windows.Forms.SizeType.Absolute
      Me.tlpInfo.ColumnStyles(2).Width = projectWidth
      Me.lblProject.Width = projectWidth
    End If
  End Sub

#End Region

#Region " Private Procedures "

  ''' <summary>
  ''' This loads the project name and client name based on the
  ''' project ID provided.
  ''' </summary>
  ''' <param name="projectID"></param>
  ''' <remarks></remarks>
  Private Sub SetupTitles(ByVal projectID As Integer)
    '-- Establish locals to contain names
    Dim projectName As String = "Sample Project"
    Dim clientName As String = "A Client"

    ''-- If the project ID is equal to the current user context
    ''   then use the current user context to get values
    'If projectID = UserContext.CurrentProjectID Then
    '  projectName = UserContext.CurrentFormattedProjectName
    '  clientName = UserContext.CurrentClientName
    'Else
    '  ' Load the indicated project's name and client name
    '  Dim clientID As Integer
    '  Using pbo As ProjectBO = New ProjectBO
    '    clientID = pbo.GetClientIDByID(projectID)
    '    projectName = pbo.GetFormattedProjectNameByID(projectID)
    '  End Using
    '  Using cbo As ClientBO = New ClientBO
    '    clientName = cbo.GetClientNameByID(clientID)
    '  End Using
    'End If

    '-- Set the project name
    Me.SetProjectName(projectName)

    '-- Set the client name. Note that an exception is thrown
    '   if there is no custom name set.
    If Me.UseCustomClientName Then
      '-- Check that a custom name has been set
      If Me.CustomClientName.Length() = 0 Then
        Throw New InvalidOperationException("The form cannot use a custom client name because no custom client name was defined.")
      End If

      '-- Use the custom client name
      Me.SetClientName(Me.CustomClientName)
    Else
      '-- Use the client associated with the client
      Me.SetClientName(clientName)
    End If
    ' Set the details text if it needs the project name
    If Me.DetailsFormatString.Trim().Length() > 0 Then
      Me.ramsHeader.DetailText = String.Format(Me.DetailsFormatString, projectName)
    End If
  End Sub

#End Region

End Class
