Imports System.ComponentModel


''' <summary>
''' Provides capability to select a vendor from within a 
''' project.
''' </summary>
''' <remarks></remarks>
Public Class SelectVendorBaseForm

#Region "   Events "

  ''' <summary>
  ''' Raised when the VendorID property is about to change.
  ''' </summary>
  ''' <remarks></remarks>
  Public Event VendorChanging As CancelEventHandler

  ''' <summary>
  ''' Raised when the VendorID property has changed.
  ''' </summary>
  ''' <remarks></remarks>
  Public Event VendorChanged As EventHandler

  ''' <summary>
  ''' Raises the VendorChanging event.
  ''' </summary>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Protected Overridable Sub OnVendorChanging(ByVal e As CancelEventArgs)
    RaiseEvent VendorChanging(Me, e)
  End Sub

  ''' <summary>
  ''' Raises the VendorChanged event.
  ''' </summary>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Protected Overridable Sub OnVendorChanged(ByVal e As EventArgs)
    RaiseEvent VendorChanged(Me, e)
  End Sub

#End Region

#Region "   Properties "

  Private _vendorTitleTemplate As String = "Vendor: {0}"

  ''' <summary>
  ''' Define the template used to build the title of the 
  ''' vendor selection box.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>
  <Category("RAMS: Vendor Area") _
  , Description("Define the template used to build the title of the vendor box. This is a String.Format template and {0} is used to place the vendor name within the title.") _
  , DefaultValue("Vendor: {0}")> _
  Public Property VendorTitleTemplate() As String
    Get
      Return _vendorTitleTemplate
    End Get
    Set(ByVal value As String)
      _vendorTitleTemplate = value
    End Set
  End Property

  Private _vendorID As Integer

  ''' <summary>
  ''' Define the ID of the vendor.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>     
  <Browsable(False)> _
  Public Property VendorID() As Integer
    Get
      Return _vendorID
    End Get
    Set(ByVal value As Integer)
      '-- Setup args for changing event
      Dim changingArgs As New CancelEventArgs()

      '-- Raise changing event
      Me.OnVendorChanging(changingArgs)

      '-- Check if the event was canceled
      If Not changingArgs.Cancel Then
        _vendorID = value

        '-- Raise the changed event
        Me.OnVendorChanged(New EventArgs())
      End If
    End Set
  End Property

  Private _vendorName As String = String.Empty

  ''' <summary>
  ''' Return the name of the selected vendor.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' When the vendor ID is set, this is looked up and set.
  ''' </remarks>
  Public ReadOnly Property VendorName() As String
    Get
      Return _vendorName
    End Get
  End Property

  Private _noVendorName As String = "<No Vendor Selected>"

  ''' <summary>
  ''' Return the name to use if there is no vendor selected.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>     
  <Category("RAMS: Vendor Area") _
  , Description("Set the text used when no vendor is selected.") _
  , DefaultValue("<No Vendor Selected>")> _
  Public Property NoVendorName() As String
    Get
      Return _noVendorName
    End Get
    Set(ByVal value As String)
      _noVendorName = value
    End Set
  End Property

#End Region

#Region "   Event Handlers "

  ''' <summary>
  ''' Set the project ID so the list of vendors can load.
  ''' </summary>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Private Sub cboSelectVendor_ListPopulating(ByVal e As MicroFour.StrataFrame.UI.ListPopulatingEventArgs) Handles cboSelectVendor.ListPopulating
    e.Parameters(0).Value = Me.ProjectID
  End Sub

  ''' <summary>
  ''' Set the vendor ID and the title of the group box.
  ''' </summary>
  ''' <param name="sender"></param>
  ''' <param name="e"></param>
  ''' <remarks></remarks>
  Private Sub cboSelectVendor_SelectedIndexChanged(ByVal sender As Object, ByVal e As System.EventArgs) Handles cboSelectVendor.SelectedIndexChanged
    '-- Get the vendor ID
    Dim id As Integer = CInt(Me.cboSelectVendor.SelectedValue)

    '-- Set the vendor ID
    Me.VendorID = id

    '-- Set the name
    Me.SetVendorName(id)

    '-- Set group box title
    Me.tcVendorSelection.Title = String.Format(Me.VendorTitleTemplate, Me.VendorName)
  End Sub

  ''' <summary>
  ''' Show the details of the current vendor.
  ''' </summary>
  ''' <remarks></remarks>
  Private Sub tcVendorSelection_CloseButtonClicked() Handles tcVendorSelection.CloseButtonClicked
    '-- Get the vendor ID
    Dim vendorID As Integer = Me.VendorID

    '-- Validate that there is a current vendor
    If vendorID = 0 Then Return

    '-- Show the vendor's details
    'FOXSystems.RAMS.Shared.FormOpener.ShowVendor(Me.VendorID)
  End Sub

#End Region

#Region "   Private (helper) Methods "

  ''' <summary>
  ''' Set the name of the vendor given the vendor's ID.
  ''' </summary>
  ''' <param name="vendorID"></param>
  ''' <remarks></remarks>
  Private Sub SetVendorName(ByVal vendorID As Integer)
    '-- Set the vendor name to the no name value. This
    '   both clears last vendor name and sets the default if no 
    '   vendor is selected or found.
    _vendorName = Me.NoVendorName

    '-- Use vendor BO to get the name. Only bother if the 
    '   vendor ID is not zero
    If vendorID > 0 Then
      'Using bo As New FOXSystems.RAMS.BusinessObjects.Project.VendorsBO()
      '  _vendorName = bo.GetVendorName(vendorID)
      'End Using
      _vendorName = "Some Vendor"
    End If
  End Sub

#End Region

End Class
