﻿''' <summary>
''' Provide formatting for boolean types.  
''' </summary>
''' <remarks>
''' The standard boolean type does not allow any formatting 
''' by default. Therefore, if a boolean value has a meaning 
''' such as "yes/no", there is no default way to format the
''' boolean.
''' 
''' Notes:
'''   - The ToString() has an overload that accepts an IFormatProvider
'''     but apparently (according to MSDN), it does nothing with that
'''     provider. This can only be used with other formatters such as
'''     String.Format().
'''   - The default formatting will return either the TrueString or the
'''     FalseString, both shared (static) methods of the Boolean type.
'''     Unfortunately, these are readonly.
''' </remarks>
Public Class BooleanFormatProvider
  Implements IFormatProvider, ICustomFormatter

#Region "   Constructors "

  ''' <summary>
  ''' Construct a standard default instance.
  ''' </summary>
  ''' <remarks></remarks>
  Public Sub New()
  End Sub

  ''' <summary>
  ''' Construc a formatter with a custom delimiter.
  ''' </summary>
  ''' <param name="delimiter"></param>
  ''' <remarks></remarks>
  Public Sub New(ByVal delimiter As Char)
    Me.Delimiter = delimiter
  End Sub

  ''' <summary>
  ''' Construct a formatter with the default indicated format.
  ''' </summary>
  ''' <param name="defaultFormat"></param>
  ''' <remarks></remarks>
  Public Sub New(ByVal defaultFormat As String)
    Me.DefaultFormat = defaultFormat
  End Sub

  ''' <summary>
  ''' Construct a formatter with a custome delimiter and a
  ''' default format string.
  ''' </summary>
  ''' <param name="defaultFormat"></param>
  ''' <param name="delimiter"></param>
  ''' <remarks></remarks>
  Public Sub New(ByVal defaultFormat As String, ByVal delimiter As Char)
    Me.Delimiter = delimiter
    Me.DefaultFormat = defaultFormat
  End Sub

#End Region

#Region "   Properties "

  Private _defaultFormat As String = "true|false"

  ''' <summary>
  ''' Define the default format used.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks>
  ''' The default is to use true and false.
  ''' 
  ''' This can be used with the ToString of a boolean:
  ''' 
  '''   True.ToString(New BooleanFormatProvider("y|n"))
  ''' </remarks>     
  Public Property DefaultFormat() As String
    Get
      Return _defaultFormat
    End Get
    Set(ByVal value As String)
      _defaultFormat = value
    End Set
  End Property

  Private _delimiter As Char = "|"c

  ''' <summary>
  ''' Define the delimiter that separates the true string
  ''' from the false string.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>     
  Public Property Delimiter() As Char
    Get
      Return _delimiter
    End Get
    Set(ByVal value As Char)
      _delimiter = value
    End Set
  End Property

#End Region

#Region "   ICustomFormatter Implementation "

  ''' <summary>
  ''' Handle the actual formatting of a boolean value.
  ''' </summary>
  ''' <param name="format1"></param>
  ''' <param name="arg"></param>
  ''' <param name="formatProvider"></param>
  ''' <returns></returns>
  ''' <remarks>
  ''' Valid formats provide two strings, one for true and one
  ''' for false. They are delimited with the character defined
  ''' for the Delimiter property, a vertical bar by default. E.g.
  ''' 
  '''   y|n
  '''   yes|no
  '''   Loaded|Unloaded
  ''' </remarks>
  Public Function Format(ByVal format1 As String, ByVal arg As Object, ByVal formatProvider As System.IFormatProvider) As String Implements System.ICustomFormatter.Format
    '-- Validate that the arg is boolean or
    If Not arg.GetType().Equals(GetType(Boolean)) Then
      Throw New ArgumentException("Requires a boolean value to format.")
    End If

    '-- If no format was provided, then use the default
    If String.IsNullOrEmpty(format1) Then
      format1 = Me.DefaultFormat
    End If

    '-- Determine if the format is likely valid, 
    '   meaning it will have a vertical bar in it.
    If format1.IndexOf(Me.Delimiter) < 0 Then
      Dim exMsg As String = String.Format("The format is not valid. The valid format is [true string{0}false string], without the square brackets.", Me.Delimiter)
      Throw New ArgumentException(exMsg, "format1")
    End If

    '-- Parse the true string and the false string
    Dim formatStrings As String() = format1.Split(Me.Delimiter)
    Dim trueString As String = formatStrings(0)
    Dim falseString As String = formatStrings(1)

    '-- Return the appropriate string
    If CType(arg, Boolean) Then
      Return trueString
    Else
      Return falseString
    End If
  End Function

#End Region

#Region "   IFormatProvider Implementation "

  ''' <summary>
  ''' Implements IFormatProvider and returns the custom formatter defined
  ''' in this class.
  ''' </summary>
  ''' <param name="formatType"></param>
  ''' <returns></returns>
  ''' <remarks></remarks>
  Public Function GetFormat(ByVal formatType As System.Type) As Object Implements System.IFormatProvider.GetFormat
    If formatType.Equals(GetType(ICustomFormatter)) Then
      Return Me
    Else
      Return Nothing
    End If
  End Function

#End Region

End Class
