﻿Imports System.ComponentModel
Imports MicroFour.StrataFrame.Messaging

Namespace Forms.Controls

  ''' <summary>
  ''' Extended ListView control that supports staying in synch 
  ''' with a BO.  
  ''' </summary>
  ''' <remarks></remarks>
  Public Class SynchListView
    Inherits MicroFour.StrataFrame.UI.Windows.Forms.ListView

#Region "   Properties "

#Region "     Public Properties "

    Private WithEvents _dataBO As MicroFour.StrataFrame.Business.BusinessLayer

    ''' <summary>
    ''' Define the BO that will be navigated.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    <Category("SynchListSample: Configure Business Object") _
    , Description("Business object that the listview stays in synch with.")> _
    Public Property DataBO() As MicroFour.StrataFrame.Business.BusinessLayer
      Get
        Return _dataBO
      End Get
      Set(ByVal value As MicroFour.StrataFrame.Business.BusinessLayer)
        _dataBO = value
        If _dataBO IsNot Nothing Then
          AddHandler _dataBO.AfterAddNew, AddressOf _dataBO_AfterAddNew
          AddHandler _dataBO.AfterDelete, AddressOf _dataBO_AfterDelete
          AddHandler _dataBO.AfterSave, AddressOf _dataBO_AfterSave
          AddHandler _dataBO.Navigating, AddressOf _dataBO_Navigating
          AddHandler _dataBO.Navigated, AddressOf _dataBO_Navigated
          AddHandler _dataBO.CurrentDataTableRefilled, AddressOf _dataBO_CurrentDataTableRefilled
        End If
      End Set
    End Property

#End Region

#Region "     Private (internallly used) Properties "

    Private _recordJustAdded As Boolean = False

    ''' <summary>
    ''' Flagged internally when the BO has just added a
    ''' new record. This is needed so the navigating event
    ''' handler can distinguish between a Dirty that is actually
    ''' dirty and one that is dirty from a brand new add.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Private Property RecordJustAdded() As Boolean
      Get
        Return _recordJustAdded
      End Get
      Set(ByVal value As Boolean)
        _recordJustAdded = value
      End Set
    End Property

    Private _recordJustDeleted As Boolean = False

    ''' <summary>
    ''' Flagged internally when the BO has just deleted a
    ''' record. This is needed so the navigating event
    ''' handler can distinguish between a Dirty that is actually
    ''' dirty and one that is dirty from being deleted.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Private Property RecordJustDeleted() As Boolean
      Get
        Return _recordJustDeleted
      End Get
      Set(ByVal value As Boolean)
        _recordJustDeleted = value
      End Set
    End Property

#End Region

#End Region

#Region "   Public Methods "

    ''' <summary>
    ''' Navigate by setting the current row of the BO.
    ''' </summary>
    ''' <param name="id">ID of PK of BO, must be an integer</param>
    ''' <remarks>
    ''' This is used to programmatically set the ID of the BO, 
    ''' based on an ID. When the BO navigates, event handlers
    ''' here will ensure that the list is in synch.
    ''' </remarks>
    Public Sub NavByBO(ByVal id As Integer)
      '-- Only do anything if the provided ID isn't already
      '   the current value.
      If id > 0 AndAlso (Me.DataBO.CurrentRowIndex < 0 OrElse CInt(Me.DataBO.Item(Me.DataBO.PrimaryKeyField)) <> id) Then
        Me.DataBO.NavigateToPrimaryKey(id)
      End If
    End Sub

    ''' <summary>
    ''' This updates the nav list and makes sure the BO
    ''' and nav list are in synch.
    ''' </summary>
    ''' <remarks>
    ''' The synching is handled by <see>SynchList</see>.
    ''' </remarks>
    Public Sub UpdateAndSynchList()
      Me.Requery()
      SynchList()
    End Sub

    ''' <summary>
    ''' Initiate a navigation using the value of the list. 
    ''' This is used when a client needs to manipulate the
    ''' list and cause a navigation.
    ''' </summary>
    ''' <remarks></remarks>
    Public Sub NavByList()
      Me.NavByBO(GetCurrentListValue())
    End Sub

    ''' <summary>
    ''' This sets the first item in the list as the selected item and then
    ''' navigates to that item in the BO.
    ''' </summary>
    ''' <remarks>
    ''' </remarks>
    Public Sub SelectFirstListItem()
      '-- Verify BO is set 
      If Me.DataBO Is Nothing Then
        Throw New InvalidOperationException("The data BO is not set.")
      End If

      '-- Verify the list and that the BO both have items
      If Me.Items.Count = 0 Then Return
      If Me.DataBO.Count = 0 Then Return

      '-- Ensure that the first item on the list is selected
      If GetNavListIndex() <> 0 Then
        Me.Items(0).Selected = True
      End If

      '-- Synch BO to this item. This simply selects the appropriate
      '   row in the BO and then lets the navigated event handle the rest.
      If CType(Me.DataBO.Item(Me.DataBO.PrimaryKeyField), Integer) <> GetCurrentListValue() Then
        Me.DataBO.NavigateToPrimaryKey(GetCurrentListValue())
      End If
    End Sub

#End Region

#Region "   Overload Methods "

    ''' <summary>
    ''' Overload of base Dispose class, so the BO can be disposed of. 
    ''' </summary>
    ''' <remarks>
    ''' </remarks>
    Public Overridable Overloads Sub Dispose()
      If _dataBO IsNot Nothing Then
        ' Remove event handlers
        RemoveHandler _dataBO.AfterAddNew, AddressOf _dataBO_AfterAddNew
        RemoveHandler _dataBO.AfterDelete, AddressOf _dataBO_AfterDelete
        RemoveHandler _dataBO.AfterSave, AddressOf _dataBO_AfterSave
        RemoveHandler _dataBO.Navigating, AddressOf _dataBO_Navigating
        RemoveHandler _dataBO.Navigated, AddressOf _dataBO_Navigated
        RemoveHandler _dataBO.CurrentDataTableRefilled, AddressOf _dataBO_CurrentDataTableRefilled

        _dataBO.Dispose()
      End If
      Me.Dispose(True)
    End Sub

#End Region

#Region "   Event Handlers "

#Region "     ListView Handlers "

    ''' <summary>
    ''' Navigate to the item selected in the list.
    ''' </summary>
    ''' <param name="sender"></param>
    ''' <param name="e"></param>
    ''' <remarks></remarks>
    Private Sub SynchListView_SelectedIndexChanged(ByVal sender As Object, ByVal e As System.EventArgs) Handles Me.SelectedIndexChanged
      Me.NavByList()
    End Sub

    ''' <summary>
    ''' Set the parameters needed to load this list with the data 
    ''' of the associated BO.
    ''' </summary>
    ''' <param name="e"></param>
    ''' <remarks></remarks>
    Private Sub SynchListView_ListPopulating(ByVal e As MicroFour.StrataFrame.UI.ListPopulatingEventArgs) Handles Me.ListPopulating
      ' Load data
      e.Parameters(0).Value = Me.DataBO
      e.Parameters(1).Value = MicroFour.StrataFrame.Business.BusinessCloneDataType.ClearAndFillFromDefaultView
    End Sub

#End Region

#Region "     BO Handlers "

    ''' <summary>
    ''' Handle setting the FK for the object, if configured. Pass
    ''' event out to any subscribers.
    ''' </summary>
    ''' <param name="e"></param>
    ''' <remarks>
    ''' If the FK properties are set, they are used. They 
    ''' only need to be set if the BO isn't participating
    ''' in a parent/child relationship.
    ''' </remarks>
    Private Sub _dataBO_AfterAddNew(ByVal e As EventArgs)
      '-- Indicate that we just added a new record
      Me.RecordJustAdded = True
    End Sub

    ''' <summary>
    ''' Set a flag so the Navigating event knows that the BO is
    ''' dirty because of a recent delete.
    ''' </summary>
    ''' <param name="e"></param>
    ''' <remarks></remarks>
    Private Sub _dataBO_AfterDelete(ByVal e As MicroFour.StrataFrame.Business.AfterDeleteEventArgs)
      Me.RecordJustDeleted = True
    End Sub

    ''' <summary>
    ''' Update the list incase the list columns displayed in the list
    ''' were changed.
    ''' </summary>
    ''' <param name="e"></param>
    ''' <remarks></remarks>
    Private Sub _dataBO_AfterSave(ByVal e As MicroFour.StrataFrame.Data.AfterSaveUndoEventArgs)
      Me.Requery()
      Me.SynchList()
    End Sub

    ''' <summary>
    ''' Check before any navigation that the current record isn't dirty.
    ''' If it is, then we prompt user to handle it. This will provide the
    ''' normal options of saving, undoing or canceling the nav. If the 
    ''' nav is canceled, then the list needs to be synched with the BO
    ''' again.
    ''' </summary>
    ''' <param name="e"></param>
    ''' <remarks>
    ''' </remarks>
    Private Sub _dataBO_Navigating(ByVal e As MicroFour.StrataFrame.Business.NavigatingEventArgs)
      '-- Handle special cases:
      '     - record is dirty and needs to be saved
      '     - Record added, list needs to be updated
      '     - Record deleted, list needs to be updated
      If e.BusinessObject.IsDirty _
                AndAlso _
          Not Me.RecordJustAdded _
                AndAlso _
          Not Me.RecordJustDeleted _
                AndAlso _
          Not Me.DataBO.BrokenRules.Count > 0 Then
        Dim title As String = "Save Changes?"
        Dim message As String = "Do you want to save changes?"
        Dim confirmMessage As New MessageItem
        confirmMessage.MessageFunctionality = MessageFunction.YesNoCancel
        confirmMessage.MessageIcon = MessagingIcon.Question
        confirmMessage.Text = message
        confirmMessage.Title = title
        Select Case MessageForm.ShowMessage(confirmMessage)
          Case MessageFunctionType.Yes
            '-- Save changes. This updates the list and synchs it
            '   (see the AfterSave event handler).
            Me.DataBO.Save()

          Case MessageFunctionType.No
            ' Undo changes
            Me.DataBO.Undo(MicroFour.StrataFrame.Business.BusinessUndoType.CurrentRowOnly)

          Case MessageFunctionType.Cancel
            ' Cancel navigation and resynch the list
            e.Cancel = True
            SynchList()
        End Select
      ElseIf Me.RecordJustDeleted OrElse Me.RecordJustAdded Then
        Me.UpdateAndSynchList()
      End If

      ' Clear the record just added/deleted flag
      If Me.RecordJustAdded Then
        Me.RecordJustAdded = False
      End If
      If Me.RecordJustDeleted Then
        Me.RecordJustDeleted = False
      End If
    End Sub

    ''' <summary>
    ''' Synch the list with the BO.
    ''' </summary>
    ''' <param name="e"></param>
    ''' <remarks></remarks>
    Private Sub _dataBO_Navigated(ByVal e As MicroFour.StrataFrame.Business.NavigatedEventArgs)
      Me.SynchList()
    End Sub

    ''' <summary>
    ''' Update the list and select the first item in the list.
    ''' </summary>
    ''' <remarks></remarks>
    Private Sub _dataBO_CurrentDataTableRefilled()
      Me.Requery()
      Me.SelectFirstListItem()
    End Sub

#End Region

#End Region

#Region "   Private (helper) Methods "

    ''' <summary>
    ''' Synch the list to the BO.
    ''' </summary>
    ''' <remarks>
    ''' This needs to check that there are any rows in the BO.
    ''' </remarks>
    Protected Sub SynchList()
      '-- Validate that a DataBO is set
      If Me.DataBO Is Nothing Then Return

      '-- Handle the case where there is no data (all items need to be cleared)
      If Me.DataBO.CurrentRowIndex < 0 Then
        Me.SelectedItems.Clear()
      Else
        '-- Get the ID of the currently selected item.
        Dim selectedID As Integer = Me.GetCurrentListValue()

        '-- Synch list to data source. This checks to see
        '   if the currently selected item is already the
        '   same item as the BO. If not, it finds the correct
        '   item and selects it.
        Dim dataID As Integer = CType(Me.DataBO.Item(Me.DataBO.PrimaryKeyField), Integer)
        If selectedID <> dataID Then
          For Each item As System.Windows.Forms.ListViewItem In Me.Items
            If CType(item.Tag, Integer) = dataID Then
              item.Selected = True

              '-- Make sure the selected item is visible
              Me.EnsureVisible(item.Index)
              Exit For
            End If
          Next
        End If
      End If
    End Sub

    ''' <summary>
    ''' Get the current value of the selected item in the list.
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Protected Function GetCurrentListValue() As Integer
      ' Check that list is set and that there are items in the list
      If Me.Items.Count = 0 OrElse Me.SelectedItems.Count = 0 Then
        Return 0
      End If

      ' Return list value
      Return CType(Me.SelectedItems(0).Tag, Integer)
    End Function

    ''' <summary>
    ''' Return the current index of the nav list. If there
    ''' is no current item, then return -1.
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Protected Function GetNavListIndex() As Integer
      If Me.SelectedItems.Count = 0 Then
        Return -1
      End If

      Return Me.SelectedIndices(0)
    End Function

#End Region

  End Class

End Namespace
